/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.classified.ads;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Data policy removing all the classified adds created by an unknown user after a delay
 */
public class ClassifiedAdsDataPolicy extends AbstractLogEnabled implements PersonalDataPolicy, Serviceable
{
    /** The content type id */
    public static final String CLASSIFIED_ADS_CONTENT_TYPE = "org.ametys.plugins.classified.ads.Content.ads";

    /** The content DAO */
    protected ContentDAO _contentDAO;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    private Period _retentionPeriod;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        
        Long config = Config.getInstance().<Long>getValue("classified-ads.data-policy.retention.period", false, null);
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo)
    {
        if (_retentionPeriod == null)
        {
            return AnonymizationResult.TOO_EARLY;
        }
        else if (userStatusInfo.getMissingSinceDate().isBefore(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            AndExpression filter = new AndExpression();
            filter.add(new UserExpression(DefaultContent.METADATA_CREATOR, Operator.EQ, userStatusInfo.getUserIdentity()));
            filter.add(new ContentTypeExpression(Operator.EQ, CLASSIFIED_ADS_CONTENT_TYPE));
            String query = ContentQueryHelper.getContentXPathQuery(filter);
            
            try (AmetysObjectIterable<Content> contents = _resolver.query(query))
            {
                if (contents.getSize() > 0)
                {
                    _contentDAO.forceDeleteContentsObj(contents.stream().toList(), null);
                    ForensicLogger.info("data.policy.gdpr.remove.classified-ads", Map.of("handled", Long.toString(contents.getSize()), "identity", userStatusInfo.getUserIdentity()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                    return AnonymizationResult.PROCESSED;
                }
                else
                {
                    return AnonymizationResult.NO_DATA;
                }
            }
            
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }

}
