/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio;

import java.io.StringWriter;
import java.util.Properties;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Node;

/**
 * Helper to import contents.
 */
public final class ContentImporterHelper
{
    private ContentImporterHelper()
    {
        // empty
    }
    
    /**
     * Transform text to docbook syntaxe
     * @param lines The paragraphs to transform
     * @return the docbook text
     */
    public static String textToDocbook (String[] lines)
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        sb.append("<article version=\"5.0\" xmlns=\"http://docbook.org/ns/docbook\">");
        
        for (String line : lines)
        {
            sb.append("<para>");
            sb.append(line.replaceAll("<", "&lt;").replaceAll("&", "&amp;").replaceAll("\r?\n", "<phrase role=\"linebreak\"/>"));
            sb.append("</para>");
        }
        
        sb.append("</article>");
        
        return sb.toString();
    }
    
    /**
     * Serialize a XML node as a String, with XML declaration and without indentation.
     * @param node the node.
     * @return the XML string.
     * @throws TransformerException if an error occurs.
     */
    public static String serializeNode(Node node) throws TransformerException
    {
        return serializeNode(node, false, false);
    }
    
    /**
     * Serialize a XML node as a String, with XML declaration.
     * @param node the node.
     * @param indent true to indent the result, false otherwise.
     * @return the XML string.
     * @throws TransformerException if an error occurs.
     */
    public static String serializeNode(Node node, boolean indent) throws TransformerException
    {
        return serializeNode(node, indent, false);
    }
    
    /**
     * Serialize a XML node as a String.
     * @param node the node.
     * @param indent true to indent the result, false otherwise.
     * @param omitXmlDeclaration true to omit XML declaration, false otherwise.
     * @return the XML string.
     * @throws TransformerException if an error occurs.
     */
    public static String serializeNode(Node node, boolean indent, boolean omitXmlDeclaration) throws TransformerException
    {
        Transformer transformer = TransformerFactory.newInstance().newTransformer();
        
        Properties format = new Properties();
        format.put(OutputKeys.METHOD, "xml");
        format.put(OutputKeys.OMIT_XML_DECLARATION, omitXmlDeclaration ? "yes" : "no");
        format.put(OutputKeys.INDENT, indent ? "yes" : "no");
        format.put(OutputKeys.ENCODING, "UTF-8");
        
        transformer.setOutputProperties(format);
        
        StringWriter writer = new StringWriter();
        DOMSource domSource = new DOMSource(node);
        StreamResult result = new StreamResult(writer);
        
        transformer.transform(domSource, result);
        
        return writer.toString();
    }
}
