/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Optional;
import java.util.stream.IntStream;

import javax.xml.transform.TransformerException;

import org.apache.commons.lang3.StringUtils;
import org.apache.xpath.XPathAPI;
import org.apache.xpath.objects.XObject;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.core.util.DateUtils;

final class DomNodeHelper
{
    private DomNodeHelper()
    {  }
    
    static String nullableStringValue(Node node, String xpath) throws TransformerException
    {
        return _nonEmptyValue(node, xpath)
                .orElse(null);
    }
    
    private static Optional<String> _nonEmptyValue(Node node, String xpath) throws TransformerException
    {
        XObject xObject = XPathAPI.eval(node, xpath);
        return Optional.of(xObject)
                .filter(o -> XObject.CLASS_NULL != o.getType())
                .map(XObject::str)
                .filter(StringUtils::isNotBlank);
    }
    
    static Date nullableDateValue(Node node, String xpath) throws TransformerException
    {
        return _nonEmptyValue(node, xpath)
                .map(dateStr -> DateTimeFormatter.ISO_LOCAL_DATE.parse(dateStr, LocalDate::from))
                .map(DateUtils::asDate)
                .orElse(null);
    }
    
    static Date nullableDatetimeValue(Node node, String xpath) throws TransformerException
    {
        return _nonEmptyValue(node, xpath)
                .map(dateStr -> DateUtils.getISODateTimeFormatter().parse(dateStr, ZonedDateTime::from))
                .map(DateUtils::asDate)
                .orElse(null);
    }
    
    static String[] stringValues(Node node, String xpath) throws TransformerException
    {
        NodeList nodeList = XPathAPI.selectNodeList(node, xpath);
        return IntStream.range(0, nodeList.getLength())
                .mapToObj(nodeList::item)
                .map(Node::getTextContent)
                .toArray(String[]::new);
    }
}
