/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.File;
import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.User;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Job for archiving data on disk, for backup or other purposes.
 */
public class ExportArchiveSchedulable extends AbstractArchiveSchedulable
{
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        User user = _getUser();
        
        String userEmail = null;
        String language = _userLanguagesManager.getDefaultLanguage();
        if (user != null)
        {
            userEmail = user.getEmail();
            
            String userLanguage = user.getLanguage();
            language = StringUtils.defaultIfBlank(userLanguage, language);
        }
        
        File output = _createFileOutput();
        
        getLogger().info("Exporting archive {} ...", output.getAbsolutePath());
        long t0 = System.currentTimeMillis();
        
        try
        {
            _archiveHandler.export(output);
            
            getLogger().info("Archive {} exported without error in {} ms", output.getAbsolutePath(), System.currentTimeMillis() - t0);
            
            // Once finished with archive, inform user
            String subject = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_MAIL_SUBJECT"), language);
            String body = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_MAIL_BODY"), language);
            
            _sendMail(subject, body + " " + output.getCanonicalPath(), userEmail);
        }
        catch (Exception e)
        {
            FileUtils.deleteQuietly(output);
            
            // Archiving encountered an error during execution, send error mail
            String subject = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_MAILERROR_SUBJECT"), language);
            String body = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_MAILERROR_BODY"), language);
            
            _sendMail(subject, body, userEmail);
            
            // rethrow to ensure the job is marked as failed
            throw e;
        }
    }
    
    private File _createFileOutput()
    {
        String timestamp = DateTimeFormatter.ofPattern("uuuuMMddHHmmss").format(ZonedDateTime.now());
        Path parent = _archiveHandler.getArchiveFolder();
        File output = parent.resolve("archive-" + timestamp + ".zip").toFile();
        return output;
    }
}
