/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.xml.transform.Templates;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamSource;

import org.apache.cocoon.components.xslt.TraxErrorListener;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.jackrabbit.commons.xml.ParsingContentHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * For {@link Mergers#IGNORE}, we need to use session.getImportContentHandler and filter the SAX events from the XML input stream
 * <br>For all &lt;sv:property sv:name="jcr:uuid" sv:type="String"&gt; and get their &lt;sv:value&gt;,
 * query the repository to check if it already exists and if it should be ignored
 * <br>And for each sax event of sv:node
 * <ol>
 * <li>if exists in the repository =&gt; ignore the node import</li>
 * <li>if does not exist in the repository =&gt; send SAX event</li>
 * </ol>
 */
public final class IgnoreMergerHelper
{
    private static final Logger __LOGGER = LoggerFactory.getLogger(IgnoreMergerHelper.class.getPackageName());
    private static final org.apache.avalon.framework.logger.Logger __AVALON_LOGGER = new SLF4JLoggerAdapter(__LOGGER);
    
    // a way to pass the Session object to the helper method #uuidIsNotPresent, which called by the XSL
    private static Map<Thread, Session> __SESSIONS_BY_REQUEST = new ConcurrentHashMap<>();
    
    private IgnoreMergerHelper()
    {  }
    
    /**
     * Tells if the given node UUID is not present in the reposiotry.
     * @param nodeUuid The node UUID to test
     * @return <code>true</code> if no node exists in the repository with this UUID. <code>false</code> if a node exists in the repository with this UUID.
     */
    public static boolean uuidIsNotPresent(String nodeUuid)
    {
        if (StringUtils.isBlank(nodeUuid))
        {
            return true;
        }
        
        try
        {
            Session session = __SESSIONS_BY_REQUEST.get(Thread.currentThread());
            Node node = session.getNodeByIdentifier(nodeUuid);
            Objects.requireNonNull(node, "Node cannot be null.");
            __LOGGER.debug("IgnoreMergerHelper#uuidIsNotPresent is called for argument '{}'. A node with this uuid was found. Returned result is false.", nodeUuid);
            return false;
        }
        catch (ItemNotFoundException e)
        {
            __LOGGER.debug("IgnoreMergerHelper#uuidIsNotPresent is called for argument '{}'. No node was found with this uuid. Returned result is true.", nodeUuid);
            return true;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    static void jcrImportXml(Session session, String parentAbsPath, InputStream in, int importUuidBehavior, Mergers mergers) throws RepositoryException, IOException
    {
        ContentHandler importContentHandler = session.getImportContentHandler(parentAbsPath, importUuidBehavior);
        Thread currentThread = Thread.currentThread();
        try
        {
            __SESSIONS_BY_REQUEST.put(currentThread, session);
            TransformerHandler filteringContentHandler = _sendFilteredSaxEvents(importContentHandler, mergers);
            _sendSaxEventsFromInputStream(in, filteringContentHandler);
        }
        catch (TransformerException | SAXException e)
        {
            throw new IOException(e);
        }
        finally
        {
            __SESSIONS_BY_REQUEST.remove(currentThread);
        }
    }
    
    private static TransformerHandler _sendFilteredSaxEvents(ContentHandler receiverContentHandler, Mergers mergers) throws IOException, TransformerConfigurationException
    {
        // Filter thanks to a XSL
        Source xsltSource = null;
        try
        {
            xsltSource = mergers._sourceResolver.resolveURI("plugin:contentio://stylesheets/archive/ignore-merger-helper.xsl");
            try (InputStream inputStream = xsltSource.getInputStream())
            {
                StreamSource xsltStreamSource = new StreamSource(inputStream);
                
                SAXTransformerFactory saxTransformerFactory = Archivers.getSaxTransformerFactory();
                Templates xsltTemplates = saxTransformerFactory.newTemplates(xsltStreamSource);
                TransformerHandler filteringTransformerHandler = saxTransformerFactory.newTransformerHandler(xsltTemplates);
                TraxErrorListener errorListener = new TraxErrorListener(__AVALON_LOGGER, xsltSource.getURI());
                filteringTransformerHandler.getTransformer().setErrorListener(errorListener);
                
                filteringTransformerHandler.setResult(new SAXResult(receiverContentHandler));
                
                return filteringTransformerHandler;
            }
        }
        finally
        {
            if (xsltSource != null)
            {
                mergers._sourceResolver.release(xsltSource);
            }
        }
    }
    
    private static void _sendSaxEventsFromInputStream(InputStream in, ContentHandler wrappedContentHandler) throws SAXException, IOException
    {
        var parsingContentHandler = new ParsingContentHandler(wrappedContentHandler);
        parsingContentHandler.parse(in);
    }
}
