/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.User;
import org.ametys.plugins.contentio.archive.ImportReport.ImportError;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * Job for importing data from an archive.
 */
public class ImportArchiveSchedulable extends AbstractArchiveSchedulable
{
    /** This schedulable id */
    public static final String ID = ImportArchiveSchedulable.class.getName();
    
    static final String ARCHIVE_KEY = "archive";
    static final String ELEMENTS_KEY = "elements";
    static final String MERGE_POLICY_KEY = "mergePolicy";
    
    private static final String __JOBDATAMAP_ARCHIVE_KEY = Scheduler.PARAM_VALUES_PREFIX + ARCHIVE_KEY;
    private static final String __JOBDATAMAP_ELEMENTS_KEY = Scheduler.PARAM_VALUES_PREFIX + ELEMENTS_KEY;
    private static final String __JOBDATAMAP_MERGE_POLICY_KEY = Scheduler.PARAM_VALUES_PREFIX + MERGE_POLICY_KEY;
    
    private static final String __ERROR_TEMPLATE = "<li><pre>%s \n %s</pre></li>";
    
    private ArchiverExtensionPoint _archiverEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _archiverEP = (ArchiverExtensionPoint) manager.lookup(ArchiverExtensionPoint.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String archiveName = jobDataMap.getString(__JOBDATAMAP_ARCHIVE_KEY);
        Optional<Collection<String>> elements = Optional.of(__JOBDATAMAP_ELEMENTS_KEY)
                .map(jobDataMap::get)
                .filter(Collection.class::isInstance)
                .map(Collection.class::cast);
        MergePolicy mergePolicy = (MergePolicy) jobDataMap.get(__JOBDATAMAP_MERGE_POLICY_KEY);
        Merger merger = mergePolicy.getMerger();
        
        User user = _getUser();
        String userEmail = null;
        String language = _userLanguagesManager.getDefaultLanguage();
        if (user != null)
        {
            userEmail = user.getEmail();
            
            String userLanguage = user.getLanguage();
            language = StringUtils.defaultIfBlank(userLanguage, language);
        }
        
        File input = _getFileInput(archiveName);
        
        getLogger().info("Importing archive {} ...", input.getAbsolutePath());
        long t0 = System.currentTimeMillis();
        
        boolean success = true;
        try
        {
            ImportReport importReport = _import(input, elements, merger);
            
            getLogger().info("Archive {} imported without error in {} ms", input.getAbsolutePath(), System.currentTimeMillis() - t0);
            
            // Once finished with import, inform user
            String subject = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUBJECT"), language);
            String body = getSuccessMailBody(input, elements, importReport, language);
            
            _sendMail(subject, body, userEmail);
            
            success = importReport.getErrors().isEmpty();
        }
        catch (Exception e)
        {
            // Importing encountered an error during execution, send error mail
            String subject = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAILERROR_SUBJECT"), language);
            String body = _i18nUtils.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAILERROR_BODY"), language);
            
            _sendMail(subject, body, userEmail);
            
            // rethrow to ensure the job is marked as failed
            throw e;
        }
        
        // throw to ensure the job is marked as failed
        if (!success)
        {
            throw new JobExecutionException("Errors occured during the import. See previous logs for more info.");
        }
    }
    
    private ImportReport _import(File input, Optional<Collection<String>> elements, Merger merger) throws IOException
    {
        if (elements.isPresent())
        {
            return _archiveHandler.partialImport(input, elements.get(), merger);
        }
        else
        {
            return _archiveHandler.importAll(input, merger);
        }
    }
    
    private File _getFileInput(String archiveName)
    {
        return _archiveHandler.getArchiveFile(archiveName);
    }
    
    /**
     * Gets the body of the success mail
     * @param input The file input
     * @param elements the imported elements or empty if all the archive was imported
     * @param importReport The {@link ImportReport}
     * @param language The language to use in the mail
     * @return the body of the success mail
     * @throws IOException If an I/O error occurs with the ZIP filesystem
     */
    protected String getSuccessMailBody(File input, Optional<Collection<String>> elements, ImportReport importReport, String language) throws IOException
    {
        final String catalogue = "plugin.contentio";
        
        Map<String, I18nizableTextParameter> i18nParameters = new HashMap<>();
        
        String inputPath = input.getCanonicalPath();
        i18nParameters.put("archivePath", new I18nizableText(inputPath));
        i18nParameters.put("archiveName", new I18nizableText(input.getName()));
        
        Collection<ImportError> errors = importReport.getErrors();
        
        StringBuilder body = new StringBuilder();
        
        if (errors.isEmpty())
        {
            body.append(_i18nUtils.translate(new I18nizableText(catalogue, "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUCCESS_BODY_INTRO", i18nParameters), language));
        }
        else if (errors.size() == 1)
        {
            body.append(_i18nUtils.translate(new I18nizableText(catalogue, "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUCCESS_BODY_INTRO_WITH_ERROR", i18nParameters), language));
        }
        else
        {
            i18nParameters.put("nbErrors", new I18nizableText(String.valueOf(errors.size())));
            body.append(_i18nUtils.translate(new I18nizableText(catalogue, "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUCCESS_BODY_INTRO_WITH_ERRORS", i18nParameters), language));
        }
        
        body.append("\n<br/><br/>")
            .append(_i18nUtils.translate(new I18nizableText(catalogue, "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUCCESS_BODY_END"), language));
        
        Predicate<Archiver> filterArchiver = elements.isPresent()
                ? archiver -> !archiver.managedPartialImports(elements.get()).isEmpty()
                : archiver -> true;
        
        // Join additional instruction
        String additionalBody = _archiverEP.getExtensionsIds()
                .stream()
                .map(_archiverEP::getExtension)
                .filter(filterArchiver)
                .map(Archiver::additionalSuccessImportMail)
                .flatMap(Collection::stream)
                .distinct()
                .map(adB -> _i18nUtils.translate(adB, language))
                .collect(Collectors.joining("<br/>"));
        
        if (!additionalBody.isBlank())
        {
            body.append("<br/>").append(additionalBody);
        }
        
        if (!errors.isEmpty())
        {
            String errorItems = errors
                    .stream()
                    .map(importError -> String.format(__ERROR_TEMPLATE, importError.getMessage(), importError.getStackTrace()))
                    .collect(Collectors.joining("\n"));
            
            body.append("\n<br/><br/>")
                .append("\n" + _i18nUtils.translate(new I18nizableText(catalogue, "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_BODY_LIST_OF_ERRORS"), language))
                .append("\n<br/>")
                .append("\n<ul>\n")
                .append(errorItems)
                .append("\n</ul>");
        }
        
        return body.toString();
    }
}
