/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;

import javax.jcr.ImportUUIDBehavior;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.excalibur.source.SourceResolver;
import org.slf4j.Logger;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The component holding the {@link Merger} implementations
 */
public class Mergers extends AbstractLogEnabled implements Component, Serviceable, Initializable, Disposable
{
    /** Avalon role. */
    public static final String ROLE = Mergers.class.getName();
    
    static final Merger DELETE_BEFORE = new DeleteBefore();
    static final Merger IGNORE = new Ignore();
    static final Merger REPLACE = new Replace();
    static final Merger FAIL = new Fail();
    
    private static final Merger[] __IMPLEMENTATIONS = new Merger[] {DELETE_BEFORE, IGNORE, REPLACE, FAIL};
    
    AmetysObjectResolver _ametysObjectResolver;
    SourceResolver _sourceResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    private static void _bindMergersComponent(Mergers mergersComponent)
    {
        for (Merger merger : __IMPLEMENTATIONS)
        {
            if (merger instanceof AbstractMerger)
            {
                ((AbstractMerger) merger).lateBindDependencies(mergersComponent);
            }
        }
    }
    
    @Override
    public void initialize() throws Exception
    {
        _bindMergersComponent(this);
    }
    
    @Override
    public void dispose()
    {
        _bindMergersComponent(null);
    }
    
    private abstract static class AbstractMerger implements Merger
    {
        /** The {@link Mergers} component to access to other components/services/dependencies */
        protected Mergers _mergers;

        void lateBindDependencies(Mergers mergersComponent)
        {
            _mergers = mergersComponent;
        }
    }
    
    // Override just to avoid warnings on usage of _mergers.getLogger()
    @Override
    protected Logger getLogger()
    {
        return super.getLogger();
    }
    
    private static final class DeleteBefore implements Merger
    {
        @Override
        public boolean deleteBeforePartialImport()
        {
            return true;
        }
        
        @Override
        public boolean needsMerge(String id)
        {
            return false;
        }

        @Override
        public AfterMerge merge(String id)
        {
            throw new NotImplementedException("Should never be called");
        }
        
        @Override
        public int getImportUuidBehavior()
        {
            return ImportUUIDBehavior.IMPORT_UUID_COLLISION_THROW; // should never have a collision, so choose here to throw
        }
    }
    
    private static final class Ignore extends AbstractMerger
    {
        @Override
        public boolean needsMerge(String id)
        {
            return _mergers._ametysObjectResolver.hasAmetysObjectForId(id);
        }

        @Override
        public AfterMerge merge(String id)
        {
            // Nothing to do
            // Indeed, we want to ignore the import of the object
            // The already existing object is kept
            // We return STOP_PROCESS so as to not import the object from the ZIP archive
            _mergers.getLogger().debug("Object for id '{}' will be ignored as it already exists ('Ignore' MergePolicy)...", id);
            return AfterMerge.STOP_PROCESS;
        }
        
        @Override
        public int getImportUuidBehavior()
        {
            // There is no behavior for that
            // So we return the safest option here, i.e. throw an error as soon as possible
            return ImportUUIDBehavior.IMPORT_UUID_COLLISION_THROW;
        }
        
        @Override
        public void jcrImportXml(Session session, String parentAbsPath, InputStream in) throws RepositoryException, IOException
        {
            IgnoreMergerHelper.jcrImportXml(session, parentAbsPath, in, getImportUuidBehavior(), _mergers);
        }
    }
    
    private static final class Replace extends AbstractMerger
    {
        @Override
        public boolean needsMerge(String id)
        {
            return _mergers._ametysObjectResolver.hasAmetysObjectForId(id);
        }

        @Override
        public AfterMerge merge(String id)
        {
            // We remove the existing object
            // Indeed, we want to replace this version by the one from the ZIP archive
            // We return CONTINUE_PROCESS so as to import the object
            AmetysObject ametysObject = _mergers._ametysObjectResolver.resolveById(id);
            if (ametysObject instanceof RemovableAmetysObject)
            {
                _mergers.getLogger().debug("Removing existing object '{}' as it will be re-created ('Replace' MergePolicy)...", ametysObject);
                ((RemovableAmetysObject) ametysObject).remove();
                return AfterMerge.CONTINUE_PROCESS;
            }
            else
            {
                throw new IllegalArgumentException(String.format(
                        "The given object (%s) for id '%s' cannot be merged with a 'Replace' MergePolicy, as the existing object is not a RemovableAmetysObject",
                        ametysObject,
                        id));
            }
        }
        
        @Override
        public int getImportUuidBehavior()
        {
            return ImportUUIDBehavior.IMPORT_UUID_COLLISION_REPLACE_EXISTING;
        }
    }
    
    private static final class Fail extends AbstractMerger
    {
        @Override
        public boolean needsMerge(String id)
        {
            return _mergers._ametysObjectResolver.hasAmetysObjectForId(id);
        }

        @Override
        public AfterMerge merge(String id) throws MergeException
        {
            AmetysObject ametysObject = _mergers._ametysObjectResolver.resolveById(id);
            throw new MergeException(String.format(
                    "The import failed because the object with id '%s' already exists and the 'Fail' MergePolicy was selected.", 
                    ametysObject,
                    id));
        }
        
        @Override
        public int getImportUuidBehavior()
        {
            return ImportUUIDBehavior.IMPORT_UUID_COLLISION_THROW;
        }
    }
}
