/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;

import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.runtime.model.type.ElementType;

/**
 * Retrieves the additional data of resources types (files, binaries, rich texts) 
 */
public class ResourcesAdditionalDataGetter implements XMLValuesExtractorAdditionalDataGetter
{
    private final Path _archivePath;
    private final Path _path;

    /**
     * Creates an additional data getter
     * @param archivePath The path to the zip archive
     * @param path The path to the object being imported
     */
    public ResourcesAdditionalDataGetter(Path archivePath, Path path)
    {
        _archivePath = archivePath;
        _path = path;
    }
    
    @Override
    public Optional<Object> getAdditionalData(String dataPath, ElementType type)
    {
        try
        {
            return _getExtractorAdditionalData(dataPath, type);
        }
        catch (Exception e)
        {
            throw new RuntimeException("An unexpected error occured", e);
        }
    }
    
    private Optional<Object> _getExtractorAdditionalData(String dataPath, ElementType type) throws Exception
    {
        switch (type.getId())
        {
            case ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID:
                return Optional.of(_getFileAdditionalData(Archivers.__FILE_ATTRIBUTES_FOLDER_NAME, dataPath));
            case ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID:
                return Optional.of(_getFileAdditionalData(Archivers.__BINARY_ATTRIBUTES_FOLDER_NAME, dataPath));
            case ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID:
                return Optional.of(_getFileAdditionalData(Archivers.__RICH_TEXT_ATTACHMENTS_FOLDER_NAME, dataPath));
            default:
                return Optional.empty();
        }
    }
    
    private Map<String, InputStream> _getFileAdditionalData(String folderName, String dataPath) throws IOException
    {
        Path zipEntryFileParentFolderPath = _path.resolve(folderName)
                .resolve(Archivers.getFolderPathFromDataPath(dataPath));
        
        String folderPath = zipEntryFileParentFolderPath.toString();
        if (ZipEntryHelper.zipEntryFolderExists(_archivePath, folderPath))
        {
            return _getFileNameAndInputStream(folderPath);
        }
        
        return Map.of();
    }
    
    private Map<String, InputStream> _getFileNameAndInputStream(String folderPath) throws IOException
    {
        DirectoryStream<Path> fileChildren = ZipEntryHelper.children(
            _archivePath, 
            Optional.of(folderPath), 
            p -> !Files.isDirectory(p));
        
        try (fileChildren)
        {
            Map<String, InputStream> files = new HashMap<>();
            
            Iterator<Path> fileIt = fileChildren.iterator();
            while (fileIt.hasNext())
            {
                Path file = fileIt.next();
                String fileName = file.getFileName().toString();
                InputStream is = ZipEntryHelper.zipEntryFileInputStream(_archivePath, file.toString());
                files.put(fileName, is);
            }
            
            return files;
        }
    }
}
