/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.in;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.dom.DOMParser;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import org.ametys.plugins.contentio.in.xml.XmlContentImporter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Component handling the import of contents, based on the {@link ContentImporter} multiple extension point.
 */
public class ContentImportManager extends AbstractLogEnabled implements Serviceable, Component
{
    
    /** The avalon role. */
    public static final String ROLE = ContentImportManager.class.getName();
    
    /** The content importer extension point. */
    protected ContentImporterExtensionPoint _contentImportEP;
    
    /** A DOM parser. */
    protected DOMParser _domParser;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _contentImportEP = (ContentImporterExtensionPoint) serviceManager.lookup(ContentImporterExtensionPoint.ROLE);
        _domParser = (DOMParser) serviceManager.lookup(DOMParser.ROLE);
    }
    
    /**
     * Import contents from a given file.
     * @param file the file describing the contents to import.
     * @return a Set of imported content IDs.
     * @throws ContentImportException if an exception occurs while importing the contents from the file.
     */
    public ImportResult importContents(File file) throws ContentImportException
    {
        String name = file.getName();
        
        if (!file.canRead() || !file.isFile())
        {
            throw new ContentImportException("The file " + name + " is not a regular file or can't be read.");
        }

        try (InputStream is = new FileInputStream(file);)
        {
            
            ContentImporter importer = getImporter(file);
            
            if (importer != null)
            {
                HashMap<String, Object> params = new HashMap<>();
                Set<String> contentIds = importer.importContents(is, params);
                
                return new ImportResult(contentIds);
            }
            else
            {
                return new ImportResult(false);
            }
        }
        catch (IOException e)
        {
            getLogger().error("IO error while importing a content from file {}", name, e);
            throw new ContentImportException("IO error while importing a content from file " + name, e);
        }
    }
    
    /**
     * Import contents from an input stream.
     * @param is an input stream on the data describing the contents to import.
     * @param name the "file" name, can be null. If it's null, some importers may not be able to identify the stream.
     * @return a Set of imported content IDs.
     * @throws ContentImportException if an exception occurs while importing the contents from the file.
     */
    public ImportResult importContents(InputStream is, String name) throws ContentImportException
    {
        File tempFile = null;
        
        try
        {
            // Create a temporary file from the input stream, to be able to read it more than once.
            if (StringUtils.isNotEmpty(name))
            {
                String baseName = FilenameUtils.getBaseName(name);
                String ext = FilenameUtils.getExtension(name);
                tempFile = File.createTempFile(baseName, ext, AmetysHomeHelper.getAmetysHomeTmp());
            }
            else
            {
                tempFile = File.createTempFile("content-import-", null, AmetysHomeHelper.getAmetysHomeTmp());
            }
            
            // Copy the stream to the temporary file.
            try (OutputStream os = new FileOutputStream(tempFile))
            {
                IOUtils.copy(is, os);
            }
            
            // Import contents from the temporary file.
            return importContents(tempFile);
        }
        catch (IOException e)
        {
            getLogger().error("IO error while importing a content from file {}", name, e);
            throw new ContentImportException("IO error while importing a content from file " + name, e);
        }
        finally
        {
            FileUtils.deleteQuietly(tempFile);
        }
    }
    
    /**
     * Get the {@link ContentImporter} supporting the file.
     * @param file the file
     * @return the {@link ContentImporter} corresponding to a 
     * @throws IOException if an exception occurs when manipulating the file
     */
    protected ContentImporter getImporter(File file) throws IOException
    {
        String name = file.getName();
        
        // Maintain an import context map, which can be reused across the calls to supports.
        Map<String, Object> importContext = new HashMap<>();
        
        // Browse importers ordered by priority: the first matching will be the one having most priority.
        for (ContentImporter importer : _contentImportEP.getImportersByPriority())
        {
            try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));)
            {
                if (supports(importer, in, name, importContext))
                {
                    return importer;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Test if an importer supports the file.
     * @param importer the importer to test.
     * @param in an InputStream on the file.
     * @param name the file name, can be null.
     * @param importContext the import context map.
     * @return true if the importer supports the file, false otherwise.
     * @throws IOException if an error occurs testing the importer.
     */
    protected boolean supports(ContentImporter importer, BufferedInputStream in, String name, Map<String, Object> importContext) throws IOException
    {
        // Specific test for XML content importers.
        if (importer instanceof XmlContentImporter)
        {
            return supportsXml((XmlContentImporter) importer, in, importContext);
        }
        
        // Standard test.
        return importer.supports(in, name);
    }
    
    /**
     * Test if an {@link XmlContentImporter} supports the file.
     * @param importer the XML content importer to test.
     * @param in an InputStream on the file.
     * @param importContext the import context map.
     * @return true if the importer supports the file, false otherwise.
     * @throws IOException if an error occurs testing the importer.
     */
    protected boolean supportsXml(XmlContentImporter importer, BufferedInputStream in, Map<String, Object> importContext) throws IOException
    {
        Document document = null;
        
        // Store the parsing status and result in the import context, to read the XML only once.
        if (!importContext.containsKey("xml-document"))
        {
            try
            {
                document = _domParser.parseDocument(new InputSource(in));
            }
            catch (Exception e)
            {
                // This is not a valid XML document: the document variable will remain null
                // and we won't try to parse the document again.
            }
            
            importContext.put("xml-document", document);
        }
        else
        {
            document = (Document) importContext.get("xml-document");
        }
        
        // If the document could be parsed, test if the importer supports it.
        if (document != null)
        {
            return importer.supports(document);
        }
        
        return false;
    }
    
    /**
     * Class representing a content import result.
     */
    public class ImportResult
    {
        
        /** If an importer supporting the file has been found. */
        protected boolean _importerFound;
        
        /** The list of imported content IDs. */
        protected Set<String> _importedContentIds;
        
        /**
         * Build an ImportResult.
         * @param importerFound true if an importer was found, false otherwise.
         */
        public ImportResult(boolean importerFound)
        {
            this._importerFound = importerFound;
            this._importedContentIds = Collections.emptySet();
        }
        
        /**
         * Build an ImportResult.
         * @param importedContentIds the imported content IDs.
         */
        public ImportResult(Collection<String> importedContentIds)
        {
            this._importerFound = true;
            this._importedContentIds = new HashSet<>(importedContentIds);
        }
        
        /**
         * Get the importerFound.
         * @return the importerFound
         */
        public boolean isImporterFound()
        {
            return _importerFound;
        }
        
        /**
         * Set the importerFound.
         * @param importerFound the importerFound to set
         */
        public void setImporterFound(boolean importerFound)
        {
            this._importerFound = importerFound;
        }
        
        /**
         * Get the importedContentIds.
         * @return the importedContentIds
         */
        public Set<String> getImportedContentIds()
        {
            return _importedContentIds;
        }
        
        /**
         * Set the importedContentIds.
         * @param importedContentIds the importedContentIds to set
         */
        public void setImportedContentIds(Collection<String> importedContentIds)
        {
            this._importedContentIds = new HashSet<>(importedContentIds);
        }
        
    }
    
}
