/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.impl.LDAPCollectionHelper.LDAPCollectionHelperSearchResult;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Implementation of {@link SynchronizableContentsCollection} to be synchronized with a LDAP data source
 */
public class LDAPSynchronizableContentsCollection extends AbstractDataSourceSynchronizableContentsCollection implements Component
{
    private static final String __PARAM_LDAP_SCOPE = "scope";
    private static final String __PARAM_LDAP_RELATIVE_DN = "peopleDN";
    private static final String __PARAM_LDAP_FILTER = "baseFilter";
    
    /** The helper for LDAP connection */
    protected LDAPCollectionHelper _ldapHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ldapHelper = (LDAPCollectionHelper) serviceManager.lookup(LDAPCollectionHelper.ROLE);
    }
        
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        // Sort is ignored in LDAP request
        
        Map<String, List<String>> mapping = getMapping();
        Map<String, Map<String, Object>> searchResults = new HashMap<>();
        
        String idField = getIdField();
        List<String> remoteKeys = mapping.get(idField);
        if (remoteKeys != null && remoteKeys.size() > 0)
        {
            // Filter from search parameters
            String filter = getFilter();
            List<String> filters = new ArrayList<>();
            if (StringUtils.isNotEmpty(filter))
            {
                filters.add(filter);
            }
            if (searchParameters != null)
            {
                for (String parameterName : searchParameters.keySet())
                {
                    filters.add(parameterName + "=" + searchParameters.get(parameterName));
                }
            }
            String filtersReduced = filters.stream().filter(StringUtils::isNotEmpty).map(s -> "(" + s + ")").reduce("", (s1, s2) -> s1 + s2);
            if (!filtersReduced.isEmpty())
            {
                filtersReduced = "(&" + filtersReduced + ")";
            }

            try
            {
                LDAPCollectionHelperSearchResult results = _ldapHelper.search(getId(), getRelativeDN(), filtersReduced, getSearchScope(), offset, limit, mapping, remoteKeys.get(0), logger, getDataSourceId());
    
                searchResults = results.searchResults();
                for (Map<String, Object> result : searchResults.values())
                {
                    Object sccValues = result.get(remoteKeys.get(0));
                    result.put(SCC_UNIQUE_ID, sccValues instanceof List<?> ? ((List<?>) sccValues).get(0) : sccValues);
                }
                
                _nbError = results.nbErrors();
                _hasGlobalError = results.hasGlobalError();
            }
            catch (Exception e)
            {
                throw new RuntimeException("An error occured when importing from LDAP UserDirectory", e);
            }
        }
        else
        {
            _nbError = 1;
            logger.error("Missing LDAP attribute in mapping for field '{}' holding the unique identifier", idField);
        }
        
        return searchResults;
    }
    
    /**
     * Get the scope for LDAP search
     * @return The scope
     */
    protected String getSearchScope()
    {
        return (String) getParameterValues().get(__PARAM_LDAP_SCOPE);
    }
    
    /**
     * Get the LDAP filter
     * @return the filter
     */
    protected String getFilter()
    {
        return (String) getParameterValues().get(__PARAM_LDAP_FILTER);
    }
    
    /**
     * Get the LDAP relative DN
     * @return the relative DN 
     */
    protected String getRelativeDN()
    {
        return (String) getParameterValues().get(__PARAM_LDAP_RELATIVE_DN);
    }
    
    @Override
    protected void configureSearchModel()
    {
        for (String columnOrCriteria : _columnsAndCriteria)
        {
            _searchModelConfiguration.addCriterion(columnOrCriteria);
            _searchModelConfiguration.addColumn(columnOrCriteria, new I18nizableText(columnOrCriteria), false);
        }
    }
    
    @Override
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, List<Object>>> organizedResults = super.getRemoteValues(searchParameters, logger);
        _ldapHelper.transformTypedAttributes(organizedResults, getContentType(), getMapping().keySet());
        return organizedResults;
    }
}
