/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.ibatis.mapping.Environment;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.type.JdbcType;
import org.apache.ibatis.type.TypeHandlerRegistry;

import org.ametys.core.datasource.AbstractMyBatisDAO;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.ZonedDateTimeTypeHandler;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.impl.typehandler.SQLBlobTypeHandler;
import org.ametys.plugins.contentio.synchronize.impl.typehandler.SQLClobTypeHandler;
import org.ametys.plugins.contentio.synchronize.impl.typehandler.SQLDoubleTypeHandler;
import org.ametys.plugins.contentio.synchronize.impl.typehandler.SQLLongTypeHandler;

/**
 * DAO for {@link SynchronizableContentsCollection}s which need to access a SQL database
 */
public class SQLCollectionDAO extends AbstractMyBatisDAO
{
    /** Avalon ROLE */
    public static final String ROLE = SQLCollectionDAO.class.getName();
    
    /** The datesource id */
    protected String _dataSourceId;
    
    @Override
    protected org.apache.ibatis.session.Configuration _getMyBatisConfiguration(Environment env)
    {
        org.apache.ibatis.session.Configuration config = super._getMyBatisConfiguration(env);
        TypeHandlerRegistry typeHandlerRegistry = config.getTypeHandlerRegistry();
        
        // Transform int types to Long
        typeHandlerRegistry.register(Object.class, JdbcType.BIGINT, SQLLongTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.INTEGER, SQLLongTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.SMALLINT, SQLLongTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.TINYINT, SQLLongTypeHandler.class);
        
        // Transform float types to Double
        typeHandlerRegistry.register(Object.class, JdbcType.FLOAT, SQLDoubleTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.NUMERIC, SQLDoubleTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.DECIMAL, SQLDoubleTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.REAL, SQLDoubleTypeHandler.class);
        
        // Transform clob types to String
        typeHandlerRegistry.register(Object.class, JdbcType.CLOB, SQLClobTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.LONGVARCHAR, SQLClobTypeHandler.class);
        
        // Transform blob types to String
        typeHandlerRegistry.register(Object.class, JdbcType.BLOB, SQLBlobTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.LONGVARBINARY, SQLBlobTypeHandler.class);

        // Transform date types to ZonedDateTime
        typeHandlerRegistry.register(Object.class, JdbcType.DATE, ZonedDateTimeTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.TIMESTAMP, ZonedDateTimeTypeHandler.class);
        
        config.setCallSettersOnNulls(true);
        
        return config;
    }
    
    /**
     * Get the list of synchronized content
     * @param params the filter paramaters
     * @param dataSourceId the datasource ID
     * @return the list of synchronized content
     */
    public List<Map<String, Object>> search(Map<String, Object> params, String dataSourceId)
    {
        _setDataSourceId(dataSourceId);
        try (SqlSession session = getSession();
             Connection connection = session.getConnection();)
        {
            Map<String, Object> sqlParams = new HashMap<>(params);
            sqlParams.put("databaseType", ConnectionHelper.getDatabaseType(connection));
            return session.selectList("ContentIO.search", sqlParams);
        }
        catch (SQLException e)
        {
            throw new IllegalStateException("A database access error occured, connection could not be closed.", e);
        }
    }
    
    /**
     * Get total count of synchronized content
     * @param params the filter paramaters
     * @param dataSourceId the datasource ID
     * @return the total count of synchronized content
     */
    public int getTotalCount(Map<String, Object> params, String dataSourceId)
    {
        _setDataSourceId(dataSourceId);
        try (SqlSession session = getSession();
             Connection connection = session.getConnection();)
        {
            Map<String, Object> sqlParams = new HashMap<>(params);
            sqlParams.put("databaseType", ConnectionHelper.getDatabaseType(connection));
            return session.selectOne("ContentIO.count", sqlParams);
        }
        catch (SQLException e)
        {
            throw new IllegalStateException("A database access error occured, connection could not be closed.", e);
        }
    }
    
    @Override
    protected void _configureDatasource(Configuration configuration) throws ConfigurationException
    {
        //Do Nothing
    }
    
    @Override
    protected String _getDataSourceId()
    {
        return _dataSourceId;
    }
    
    /**
     * Set the datasource id
     * @param dataSourceId the datasource id
     */
    protected void _setDataSourceId(String dataSourceId)
    {
        this._dataSourceId = dataSourceId;
    }
    
}
