/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.search.query;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.ametys.cms.search.query.NotQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.systemprop.CollectionsSystemProperty;

/**
 * {@link Query} testing {@link SynchronizableContentsCollection}s of contents.
 */
public class CollectionsQuery implements Query
{
    private Operator _operator;
    private List<String> _collectionIds;
    
    /**
     * Build a CollectionsQuery.
     * @param ids The ids of the {@link SynchronizableContentsCollection}s to test
     */
    public CollectionsQuery(String... ids)
    {
        this(Operator.EQ, ids);
    }
    
    /**
     * Build a CollectionsQuery.
     * @param ids The ids of the {@link SynchronizableContentsCollection}s to test
     */
    public CollectionsQuery(Collection<String> ids)
    {
        this(Operator.EQ, ids);
    }
    
    /**
     * Build a CollectionsQuery.
     * @param operator The query operator (can be EQ or NE).
     * @param ids The ids of the {@link SynchronizableContentsCollection}s to test
     */
    public CollectionsQuery(Operator operator, String... ids)
    {
        this(operator, Arrays.asList(ids));
    }
    
    /**
     * Build a CollectionsQuery.
     * @param operator The query operator (can be EQ or NE).
     * @param ids The ids of the {@link SynchronizableContentsCollection}s to test
     */
    public CollectionsQuery(Operator operator, Collection<String> ids)
    {
        if (Operator.EQ != operator && Operator.NE != operator)
        {
            throw new IllegalArgumentException("Test operator '" + operator + "' is unknown for test's expression.");
        }
        
        _operator = operator;
        _collectionIds = new ArrayList<>(ids);
    }
    
    @Override
    public String build() throws QuerySyntaxException
    {
        StringBuilder query = new StringBuilder();
        
        if (_operator == Operator.NE)
        {
            NotQuery.appendNegation(query).append('(');
        }
        
        boolean first = true;
        for (String collectionId : _collectionIds)
        {
            if (!first)
            {
                query.append(" OR ");
            }
            query.append('(').append(CollectionsSystemProperty.SOLR_FIELD_NAME).append(':').append(collectionId).append(')');
            first = false;
        }
        
        if (_operator == Operator.NE)
        {
            query.append(')');
        }
        
        return query.toString();
    }
}
