/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.lang3.StringUtils;

/**
 * Utility class to retrieve {@link Path} representing folders or files under a ZIP archive.
 */
public final class ZipEntryHelper
{
    private ZipEntryHelper()
    { /*empty*/ }
    
    private static FileSystem _zipFileSystem(Path zipArchivePath) throws IOException
    {
        return FileSystems.newFileSystem(zipArchivePath, ClassLoader.getSystemClassLoader());
    }
    
    /**
     * Return the {@link Path} of the root of the given ZIP archive.
     * @param zipArchivePath The zip path
     * @return the PAth of the root
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static Path zipFileRoot(Path zipArchivePath) throws IOException
    {
        FileSystem zipFileSystem = _zipFileSystem(zipArchivePath);
        Path[] roots = StreamSupport.stream(zipFileSystem.getRootDirectories().spliterator(), false)
                .toArray(Path[]::new);
        if (roots.length != 1)
        {
            throw new IllegalStateException(String.format("Unexpected error, the zip file '%s' should have 1 and only 1 root directory", zipArchivePath));
        }
        
        Path zipFileRoot = roots[0];
        return zipFileRoot;
    }
    
    /**
     * Returns a Stream that is lazily populated with Path by searching for files in a file tree rooted at a given starting path within the ZIP, or the ZIP root. 
     * @param zipArchivePath The zip path
     * @param startingPath The (optional) starting path within the ZIP
     * @param filter The filter
     * @return the {@link Stream} of {@link Path}
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static Stream<Path> zipFileTree(Path zipArchivePath, Optional<String> startingPath, BiPredicate<Path, BasicFileAttributes> filter) throws IOException
    {
        return _zipFileTree(zipArchivePath, startingPath, Integer.MAX_VALUE, filter);
    }
    
    private static Stream<Path> _zipFileTree(Path zipArchivePath, Optional<String> startingPath, int maxDepth, BiPredicate<Path, BasicFileAttributes> filter) throws IOException
    {
        Path zipFileRoot = zipFileRoot(zipArchivePath);
        Path start = startingPath
                .map(zipFileRoot::resolve)
                .orElse(zipFileRoot);
        return Files.find(start, maxDepth, filter);
    }
    
    /**
     * Opens a directory, returning a {@link DirectoryStream} to iterate over the entries in the directory.
     * @param zipArchivePath The zip path
     * @param startingPath The (optional) starting path within the ZIP
     * @param filter The filter
     * @return a new and opened {@link DirectoryStream} object
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static DirectoryStream<Path> children(Path zipArchivePath, Optional<String> startingPath, DirectoryStream.Filter<? super Path> filter) throws IOException
    {
        Path zipFileRoot = zipFileRoot(zipArchivePath);
        Path start = startingPath
                .map(zipFileRoot::resolve)
                .orElse(zipFileRoot);
        return Files.newDirectoryStream(start, filter);
    }
    
    /**
     * Checks if the given folder entry exists in the ZIP archive.
     * @param zipPath The zip path
     * @param zipEntryFolderPath The path of the folder within the ZIP
     * @return <code>true</code> if the entry exists
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static boolean zipEntryFolderExists(Path zipPath, String zipEntryFolderPath) throws IOException
    {
        Path zipEntryPath = _zipFileSystem(zipPath)
                .getPath(zipEntryFolderPath);
        return Files.exists(zipEntryPath) && Files.isDirectory(zipEntryPath);
    }
    
    /**
     * Checks if the given file entry exists in the ZIP archive.
     * @param zipPath The zip path
     * @param zipEntryFilePath The path of the file within the ZIP
     * @return <code>true</code> if the entry exists
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static boolean zipEntryFileExists(Path zipPath, String zipEntryFilePath) throws IOException
    {
        Path zipEntryPath = _zipFileSystem(zipPath)
                .getPath(zipEntryFilePath);
        return Files.exists(zipEntryPath) && Files.isRegularFile(zipEntryPath);
    }
    
    /**
     * Returns an input stream for reading the content of an entry file of a ZIP archive.
     * <br>You may check if the entry exist with {@link #zipEntryFileExists}, otherwise you may get an {@link IOException}.
     * <br>It must be closed.
     * @param zipPath The zip path
     * @param zipEntryFilePath The path within the ZIP
     * @return a new input stream
     * @throws IOException if an I/O error is thrown when accessing a file.
     */
    public static InputStream zipEntryFileInputStream(Path zipPath, String zipEntryFilePath) throws IOException
    {
        var zipFile = new ZipFile(zipPath.toFile());
        String name = StringUtils.stripStart(zipEntryFilePath, "/");
        ZipEntry zipEntry = zipFile.getEntry(name);
        if (zipEntry == null)
        {
            throw new IOException(String.format("Wrong zip entry file path '%s', it cannot be found in the archive '%s'", zipEntryFilePath, zipPath));
        }
        
        InputStream manifestIs = zipFile.getInputStream(zipEntry);
        return new FilterInputStream(manifestIs)
        {
            // according to java.util.zip.ZipFile.getInputStream(ZipEntry),
            // "Closing this ZIP file will, in turn, close all input streams that have been returned by invocations of this method"
            // thus we need to override the close method of the returned InputStream in order to close the zip file
            @Override
            public void close() throws IOException
            {
                super.close();
                zipFile.close();
            }
        };
    }
}

