/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.plugins.contentio.synchronize.search.SCCSearchModelConfiguration;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemAccessor;

/**
 * Configuration is separated from {@link AbstractSynchronizableContentsCollection}
 */
public abstract class AbstractStaticSynchronizableContentsCollection implements SynchronizableContentsCollection, Configurable, Serviceable
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    /** The content DAO */
    protected ContentDAO _contentDAO;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The id */
    protected String _id;
    /** The label */
    protected I18nizableText _label;
    /** The path to the metadata holding the 'restricted' property */
    protected String _restrictedField;
    /** The handled content type */
    protected String _contentType;
    /** The handled languages */
    protected List<String> _languages;
    /** The id of controller */
    protected String _modelId;
    /** The untyped values of controller's parameters */
    protected Map<String, Object> _modelParamValues;
    /** True if removal sync */
    protected boolean _removalSync;
    /** True to ignore restrictions on attributes while synchronizing */
    protected boolean _ignoreRestrictions;
    /** The name of the workflow */
    protected String _workflowName;
    /** The id of the initial action of the workflow */
    protected int _initialActionId;
    /** The id of the synchronize action of the workflow */
    protected int _synchronizeActionId;
    /** The id of the validate action of the workflow */
    protected int _validateActionId;
    /** The prefix of the contents */
    protected String _contentPrefix;
    /** True to validate contents after import */
    protected boolean _validateAfterImport;
    /** The report mails */
    protected String _reportMails;
    /** The id of the content operator to use */
    protected String _synchronizingContentOperator;
    /** The id of the content operator to use */
    protected boolean _synchronizeExistingContentsOnly;
    /** <code>true</code> to check the collection while synchronizing */
    protected boolean _checkCollection;
    /** List of compatible SCC with the current SCC (excepting the current one) */
    protected List<String> _compatibleSCC;
    /** Search model configuration for search tool */
    protected SCCSearchModelConfiguration _searchModelConfiguration;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        configureStaticParams(configuration);
        configureDataSource(configuration);
        _searchModelConfiguration = new SCCSearchModelConfiguration();
        configureSearchModel();
    }

    /**
     * Called in {@link #configure(Configuration)} for first configurations needed.
     * @param configuration Configuration to read
     * @throws ConfigurationException If an error occurs
     */
    protected void configureStaticParams(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), null);
        _contentType = configuration.getChild("contentType").getValue();
        _removalSync = configuration.getChild("removalSync").getValueAsBoolean(false);
        _ignoreRestrictions = configuration.getChild("ignoreRestrictions").getValueAsBoolean(true);
        _workflowName = configuration.getChild("workflowName").getValue();
        _initialActionId = configuration.getChild("initialActionId").getValueAsInteger();
        _synchronizeActionId = configuration.getChild("synchronizeActionId").getValueAsInteger();
        _validateActionId = configuration.getChild("validateActionId").getValueAsInteger();
        _contentPrefix = configuration.getChild("contentPrefix").getValue();
        _restrictedField = configuration.getChild("restrictedField").getValue(null);
        _validateAfterImport = configuration.getChild("validateAfterImport").getValueAsBoolean(false);
        _reportMails = configuration.getChild("reportMails").getValue("");
        _synchronizingContentOperator = configuration.getChild("contentOperator").getValue();
        _modelId = configuration.getChild("model").getAttribute("id");
        _languages = _parseMultipleValuesConf(configuration.getChild("languages"));
        _modelParamValues = _parseParameters(configuration.getChild("model"));
        _synchronizeExistingContentsOnly = configuration.getChild("synchronizeExistingContentsOnly").getValueAsBoolean(false);
        _checkCollection = configuration.getChild("checkCollection").getValueAsBoolean(true);
        _compatibleSCC = _parseMultipleValuesConf(configuration.getChild("compatibleSCC"));
    }

    /**
     * Configure the data source parameters.
     * @param configuration Configuration to read
     * @throws ConfigurationException If an error occurs
     */
    protected abstract void configureDataSource(Configuration configuration) throws ConfigurationException;
    
    /**
     * Configure the search model used by SCCSearchTool.
     */
    protected abstract void configureSearchModel();
    
    public String getId()
    {
        return _id;
    }
    
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    public String getContentType()
    {
        return _contentType;
    }
    
    public List<String> getLanguages()
    {
        return _languages;
    }
    
    public String getRestrictedField()
    {
        return _restrictedField;
    }
    
    public String getSynchronizeCollectionModelId()
    {
        return _modelId;
    }
    
    public Map<String, Object> getParameterValues()
    {
        return _modelParamValues;
    }
    
    public boolean removalSync()
    {
        return _removalSync;
    }
    
    public boolean ignoreRestrictions()
    {
        return _ignoreRestrictions;
    }
    
    public boolean checkCollection()
    {
        return _checkCollection;
    }
    
    public List<String> getCompatibleSCC(boolean includeCurrent)
    {
        if (includeCurrent)
        {
            List<String> compatibleSCC = new ArrayList<>(_compatibleSCC);
            compatibleSCC.add(getId());
            return compatibleSCC;
        }
        return _compatibleSCC;
    }
    
    public String getWorkflowName()
    {
        return _workflowName;
    }
    
    public int getInitialActionId()
    {
        return _initialActionId;
    }
    
    public int getSynchronizeActionId()
    {
        return _synchronizeActionId;
    }
    
    public int getValidateActionId()
    {
        return _validateActionId;
    }
    
    public String getContentPrefix()
    {
        return _contentPrefix;
    }
    
    public boolean validateAfterImport()
    {
        return _validateAfterImport;
    }
    
    public String getReportMails()
    {
        return _reportMails;
    }
    
    public String getSynchronizingContentOperator()
    {
        return _synchronizingContentOperator;
    }
    
    public boolean synchronizeExistingContentsOnly()
    {
        return _synchronizeExistingContentsOnly;
    }
    
    public SCCSearchModelConfiguration getSearchModelConfiguration()
    {
        return _searchModelConfiguration;
    }

    /**
     * Parse parameters' values
     * @param configuration The root configuration
     * @return The parameters
     * @throws ConfigurationException if an error occurred
     */
    protected Map<String, Object> _parseParameters(Configuration configuration) throws ConfigurationException
    {
        Map<String, Object> values = new LinkedHashMap<>();
        
        Configuration[] params = configuration.getChildren("param");
        for (Configuration paramConfig : params)
        {
            values.put(paramConfig.getAttribute("name"), paramConfig.getValue(""));
        }
        return values;
    }
    
    /**
     * Parse multiple values configuration
     * @param configuration the configuration
     * @return the list of handled values
     * @throws ConfigurationException if an error occurred
     */
    protected List<String> _parseMultipleValuesConf(Configuration configuration) throws ConfigurationException
    {
        List<String> values = new ArrayList<>();
        for (Configuration conf : configuration.getChildren("value"))
        {
            values.add(conf.getValue());
        }
        return values;
    }
    
    /**
     * Transform the remote values to take each attribute cardinality into account
     * @param remoteValues the remote values
     * @param contentTypeId the content type ID from which attributes come from
     * @return the transformed values
     */
    protected Map<String, Object> _transformRemoteValuesCardinality(Map<String, List<Object>> remoteValues, String contentTypeId)
    {
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        return _transformRemoteValuesCardinality(remoteValues, contentType);
    }
    
    @SuppressWarnings("unchecked")
    private Map<String, Object> _transformRemoteValuesCardinality(Map<String, List<Object>> remoteValues, ModelItemAccessor modelItemAccessor)
    {
        Map<String, Object> transformedContentValues = new HashMap<>();
        for (String attributeName : remoteValues.keySet())
        {
            if (modelItemAccessor.hasModelItem(attributeName))
            {
                List<Object> attributeValues = remoteValues.get(attributeName);
                Object transformedAttributeValue = attributeValues;
                
                ModelItem modelItem = modelItemAccessor.getModelItem(attributeName);
                if (modelItem instanceof ElementDefinition definition && !definition.isMultiple())
                {
                    transformedAttributeValue = attributeValues.stream()
                                                               .filter(Objects::nonNull)
                                                               .findFirst()
                                                               .orElse(null);
                }
                else if (modelItem instanceof CompositeDefinition composite)
                {
                    transformedAttributeValue = attributeValues.stream()
                                                               .filter(Objects::nonNull)
                                                               .findFirst()
                                                               .map(object -> (Map<String, List<Object>>) object)
                                                               .map(values -> _transformRemoteValuesCardinality(values, composite))
                                                               .orElse(null);
                }
                else if (modelItem instanceof RepeaterDefinition repeater)
                {
                    transformedAttributeValue = attributeValues.stream()
                                                               .filter(Objects::nonNull)
                                                               .map(object -> (Map<String, List<Object>>) object)
                                                               .map(values -> _transformRemoteValuesCardinality(values, repeater))
                                                               .toList();
                }
                
                transformedContentValues.put(attributeName, transformedAttributeValue);
            }
        }
        
        return transformedContentValues;
    }
}
