/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.contentio.synchronize.AbstractSimpleSynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;

/**
 * Abstract implementation of {@link SynchronizableContentsCollection} to be synchronized with a data source
 */
public abstract class AbstractDataSourceSynchronizableContentsCollection extends AbstractSimpleSynchronizableContentsCollection
{
    /** Name of parameter holding the data source id */
    protected static final String __PARAM_DATASOURCE_ID = "datasourceId";
    /** Name of parameter holding the data source id */
    protected static final String __PARAM_FIELD_ID = "idField";
    /** Name of parameter holding the fields mapping */
    protected static final String __PARAM_MAPPING = "mapping";
    /** Name of parameter into mapping holding the synchronized property */
    protected static final String __PARAM_MAPPING_SYNCHRO = "synchro";
    /** Name of parameter into mapping holding the path of metadata */
    protected static final String __PARAM_MAPPING_METADATA_REF = "metadata-ref";
    /** Name of parameter into mapping holding the remote attribute */
    protected static final String __PARAM_MAPPING_ATTRIBUTE = "attribute";
    
    /** The JSON Utils */
    protected JSONUtils _jsonUtils;
    
    /** Mapping of the metadata with source data */
    protected Map<String, List<String>> _mapping;
    /** Columns and criteria for search */
    protected Set<String> _columnsAndCriteria;
    /** Synchronized fields */
    protected Set<String> _syncFields;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public String getIdField()
    {
        return (String) getParameterValues().get(__PARAM_FIELD_ID);
    }
    
    /**
     * Get the id of data source
     * @return The id of data source
     */
    public String getDataSourceId()
    {
        return (String) getParameterValues().get(__PARAM_DATASOURCE_ID);
    }
    
    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        _mapping = new HashMap<>();
        _columnsAndCriteria = new TreeSet<>();
        _syncFields = new HashSet<>();
        String mappingAsString = (String) getParameterValues().get(__PARAM_MAPPING);
        if (StringUtils.isNotEmpty(mappingAsString))
        {
            List<Object> mappingAsList = _jsonUtils.convertJsonToList(mappingAsString);
            for (Object object : mappingAsList)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> field = (Map<String, Object>) object;
                String metadataRef = (String) field.get(__PARAM_MAPPING_METADATA_REF);
                String[] mappingAttributes = ((String) field.get(__PARAM_MAPPING_ATTRIBUTE)).split(",");
                List<String> attributes = Stream.of(mappingAttributes)
                                                .map(String::trim)
                                                .collect(Collectors.toList());
                
                _mapping.put(metadataRef, attributes);
                for (String attribute : attributes)
                {
                    _columnsAndCriteria.add(attribute);
                }

                boolean isSynchronized = field.containsKey(__PARAM_MAPPING_SYNCHRO) ? (Boolean) field.get(__PARAM_MAPPING_SYNCHRO) : false;
                if (isSynchronized)
                {
                    _syncFields.add(metadataRef);
                }
            }
        }
    }
    
    @Override
    protected void configureSearchModel()
    {
        for (String columnOrCriteria : _columnsAndCriteria)
        {
            _searchModelConfiguration.addCriterion(columnOrCriteria);
            _searchModelConfiguration.addColumn(columnOrCriteria);
        }
    }

    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        return _syncFields;
    }
    
    /**
     * Get the field mapping
     * @return The mapping
     */
    public Map<String, List<String>> getMapping()
    {
        return _mapping;
    }
    
    @Override
    protected Map<String, Object> putIdParameter(String idValue)
    {
        Map<String, Object> parameters = new HashMap<>();

        List<String> remoteKeys = getMapping().get(getIdField());
        if (remoteKeys != null && remoteKeys.size() > 0)
        {
            parameters.put(remoteKeys.get(0), idValue);
        }
        return parameters;
    }
    
    @Override
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, Object>> results = internalSearch(searchParameters, 0, Integer.MAX_VALUE, null, logger);
        return _sccHelper.organizeRemoteValuesByAttribute(results, getMapping());
    }
}
