/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.slf4j.Logger;

import org.ametys.plugins.contentio.synchronize.AbstractSimpleSynchronizableContentsCollection;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * <p>Abstract for default and simplified implementation of SCC.</p>
 * <p>Mandatory methods are:</p>
 * <ul>
 *      <li>{@link #getIdField()} to define the content field which is the synchronization identifier.</li>
 *      <li>{@link #internalSearch(Map, int, int, List, Logger)} to search data. It supports pagination but on single search and global synchronization, pagination is set to 0 and {@link Integer#MAX_VALUE}. Also, on an import or synchronization (single search), the search parameters are filled with the {@link #getIdField()} as a parameter and its associated value. The values are organized by ID field then name and value, the value can be a {@link Collection}. May not return null.</li>
 * </ul>
 * <p>It may be useful to override the following methods too:</p>
 * <ul>
 *      <li>{@link #getLocalAndExternalFields(Map)} to set the values from synchronization into remote values. These attributes name will have the synchronization toggle on edition.</li>
 *      <li>{@link #_getMapping(Map)} to modify the default mapping, default mapping consider the field name from results as the final attribute name. The mapping can receive data from several fields.</li>
 * </ul>
 * <p>You have to declare your now SCC model into a plugin.xml through the following extension:</p>
 * <pre>
 *  &lt;extension id="fr.ametys.myproject.scc.static"
 *               class="org.ametys.plugins.contentio.synchronize.impl.DefaultSynchronizableContentsCollectionModel"
 *               point="org.ametys.plugins.contentio.synchronize.SynchronizeContentsCollectionModelExtensionPoint"&gt;
 *      &lt;class name="org.ametys.plugins.contentio.synchronize.impl.StaticSynchronizableContentsCollection"/&gt;
 *          &lt;label i18n="false"&gt;Static SCC&lt;/label&gt;
 *          &lt;description i18n="false"&gt;Static SCC&lt;/description&gt;
 *  &lt;/extension&gt;
 * </pre>
 * <p>And it's also possible to declare a synchronization button on the content:</p>
 * <pre>
 *  &lt;extension id="fr.ametys.myproject.scc.static.SynchronizePerson"
 *                point="org.ametys.core.ui.RibbonControlsManager"
 *                class="org.ametys.plugins.contentio.synchronize.clientsideelement.SCCSmartContentClientSideElement"&gt;
 *      &lt;class name="Ametys.plugins.cms.content.controller.SmartContentController"&gt;
 *          &lt;action&gt;Ametys.plugins.contentio.search.SynchronizeContentAction.act&lt;/action&gt;
 * 
 *          &lt;sccModelId&gt;fr.ametys.myproject.scc.static&lt;/sccModelId&gt;
 * 
 *          &lt;label type="false"&gt;Synchroniser le contenu&lt;/label&gt;
 *          &lt;description type="false"&gt;Synchronisation du contenu&lt;/description&gt;
 * 
 *          &lt;field-label&gt;Login&lt;/field-label&gt;
 * 
 *          &lt;selection-target-id&gt;^content$&lt;/selection-target-id&gt;
 *          &lt;selection-target-parameter&gt;
 *              &lt;name&gt;^types$&lt;/name&gt;
 *              &lt;value&gt;^org.ametys.plugins.odf.Content.person$&lt;/value&gt;
 *          &lt;/selection-target-parameter&gt;
 *          &lt;selection-enable-multiselection&gt;false&lt;/selection-enable-multiselection&gt;
 * 
 *          &lt;icon-glyph&gt;ametysicon-arrow123&lt;/icon-glyph&gt;
 * 
 *          &lt;selection-description-empty type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_NOCONTENT&lt;/selection-description-empty&gt;
 *          &lt;selection-description-nomatch type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_NOCONTENT&lt;/selection-description-nomatch&gt;
 *          &lt;selection-description-multiselectionforbidden type="i18n"&gt;plugin.cms:CONTENT_EDIT_DESCRIPTION_MANYCONTENT&lt;/selection-description-multiselectionforbidden&gt;
 *          
 *          &lt;enabled-on-right-only&gt;true&lt;/enabled-on-right-only&gt;
 *          &lt;rights&gt;RIGHT_ID&lt;/rights&gt;
 *          &lt;noright-start-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_NORIGHT_START&lt;/noright-start-description&gt;
 *          &lt;noright-end-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_NORIGHT_END&lt;/noright-end-description&gt;
 *          &lt;noright-content-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_NORIGHT_CONTENT&lt;/noright-content-description&gt;
 * 
 *          &lt;allright-start-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_START&lt;/allright-start-description&gt;
 *          &lt;allright-end-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_CONTENT_END&lt;/allright-end-description&gt;
 *          &lt;allright-content-description type="i18n"&gt;plugin.contentio:PLUGINS_CONTENTIO_BUTTON_SYNCHRONIZE_CONTENT&lt;/allright-content-description&gt;
 *          &lt;error-description type="i18n"&gt;plugin.cms:CONTENT_EDIT_DESCRIPTION_ERROR&lt;/error-description&gt;
 * 
 *          &lt;enabled-on-unlock-only&gt;true&lt;/enabled-on-unlock-only&gt;
 *          &lt;locked-start-description type="i18n"&gt;plugin.cms:CONTENT_EDIT_DESCRIPTION_LOCKED_START&lt;/locked-start-description&gt;
 *          &lt;locked-end-description type="i18n"&gt;plugin.cms:CONTENT_EDIT_DESCRIPTION_LOCKED_END&lt;/locked-end-description&gt;
 *          &lt;locked-content-description type="i18n"&gt;plugin.cms:CONTENT_EDIT_DESCRIPTION_LOCKED_CONTENT&lt;/locked-content-description&gt;
 *      &lt;/class&gt;
 *      &lt;scripts&gt;
 *          &lt;file plugin="cms"&gt;js/Ametys/plugins/cms/content/controller/SmartContentController.js&lt;/file&gt;
 *          &lt;file plugin="contentio"&gt;js/Ametys/plugins/contentio/search/SynchronizeContentAction.js&lt;/file&gt;
 *      &lt;/scripts&gt;
 *      &lt;depends&gt;
 *          &lt;org.ametys.core.ui.UIToolsFactoriesManager&gt;uitool-server-logs&lt;/org.ametys.core.ui.UIToolsFactoriesManager&gt;
 *      &lt;/depends&gt;
 *  &lt;/extension&gt;
 * </pre>
 * <p>And a tool to search and import a single content:</p>
 * <pre>
 *  &lt;extension id="fr.ametys.myproject.scc.static.Import"
 *                point="org.ametys.core.ui.RibbonControlsManager"
 *                class="org.ametys.plugins.contentio.synchronize.clientsideelement.SCCClientSideElement"&gt;
 *      &lt;class name="Ametys.ribbon.element.ui.button.OpenToolButtonController"&gt;
 *          &lt;opentool-id&gt;uitool-scc-import&lt;/opentool-id&gt;
 *          &lt;opentool-params&gt;
 *              &lt;controllerId&gt;fr.ametys.myproject.scc.static.Import&lt;/controllerId&gt;
 *          &lt;/opentool-params&gt;
 * 
 *          &lt;sccModelId&gt;fr.ametys.myproject.scc.static&lt;/sccModelId&gt;
 * 
 *          &lt;label type="false"&gt;Importer un contenu&lt;/label&gt;
 *          &lt;description type="false"&gt;Import d'un contenu&lt;/description&gt;
 * 
 *          &lt;icon-glyph&gt;ametysicon-body-people&lt;/icon-glyph&gt;
 *          &lt;icon-decorator&gt;decorator-ametysicon-upload119&lt;/icon-decorator&gt;
 *          &lt;icon-decorator-type&gt;action-create&lt;/icon-decorator-type&gt;
 *      &lt;/class&gt;
 *      &lt;depends&gt;
 *          &lt;org.ametys.core.ui.UIToolsFactoriesManager&gt;uitool-scc-import&lt;/org.ametys.core.ui.UIToolsFactoriesManager&gt;
 *      &lt;/depends&gt;
 *  &lt;/extension&gt;
 * </pre>
 * <p>The feature containing this tool must depend on contentio/org.ametys.plugins.contentio.scc.search.tool feature</p>
 */
public abstract class AbstractDefaultSynchronizableContentsCollection extends AbstractSimpleSynchronizableContentsCollection
{
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        // By default, no synchronizable fields
        return new HashSet<>();
    }

    @Override
    protected Map<String, Object> putIdParameter(String syncCode)
    {
        Map<String, Object> parameters = new HashMap<>();
        // Set the current content in search
        parameters.put(getIdField(), syncCode);
        return parameters;
    }
    
    @Override
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, Object>> results = internalSearch(searchParameters, 0, Integer.MAX_VALUE, null, logger);
        return _sccHelper.organizeRemoteValuesByAttribute(results, _getMapping(results));
    }
    
    @Override
    public Map<String, Map<String, Object>> search(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        Map<String, Map<String, Object>> results = new HashMap<>(super.search(searchParameters, offset, limit, sort, logger));
        
        // Add SCC unique ID for search results
        for (Entry<String, Map<String, Object>> result : results.entrySet())
        {
            Map<String, Object> resultValues = new HashMap<>(result.getValue());
            if (!resultValues.containsKey(SCC_UNIQUE_ID))
            {
                resultValues.put(SCC_UNIQUE_ID, result.getKey());
                results.put(result.getKey(), resultValues);
            }
        }
        
        return results;
    }
    
    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        // Do nothing
    }

    @Override
    protected void configureSearchModel()
    {
        _searchModelConfiguration.addColumn(SCC_UNIQUE_ID, new I18nizableText(getIdField()), false);
    }
    
    /**
     * Get the mapping of the current SCC based on results. Default implementation is key -> List.of(key).
     * @param results The results
     * @return the mapping based on results (if needed)
     */
    protected Map<String, List<String>> _getMapping(Map<String, Map<String, Object>> results)
    {
        // Get all columns names and organize it as mapping
        // Default implementation is key -> List.of(key) and add the synchronization code mapping too if not in the results list
        Map<String, List<String>> mapping = results.values()
                .stream()
                .map(Map::keySet)
                .flatMap(Set::stream)
                .distinct()
                .collect(Collectors.toMap(Function.identity(), key -> List.of(key)));
        mapping.putIfAbsent(getIdField(), List.of(getIdField()));
        return mapping;
    }
}
