/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl.typehandler;

import java.io.IOException;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.commons.io.IOUtils;
import org.apache.ibatis.type.BaseTypeHandler;
import org.apache.ibatis.type.JdbcType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.CharMatcher;

/**
 * Handle Clob SQL type to return string value
 */
public class SQLClobTypeHandler extends BaseTypeHandler<Object>
{
    private static Logger __logger = LoggerFactory.getLogger(SQLClobTypeHandler.class);
    
    @Override
    public void setNonNullParameter(PreparedStatement ps, int i, Object parameter, JdbcType jdbcType) throws SQLException
    {
        ps.setObject(i, parameter);
    }

    @Override
    public Object getNullableResult(ResultSet rs, String columnName) throws SQLException
    {
        Clob clob = rs.getClob(columnName);
        return _transformClobToString(clob);
    }

    @Override
    public Object getNullableResult(ResultSet rs, int columnIndex) throws SQLException
    {
        Clob clob = rs.getClob(columnIndex);
        return _transformClobToString(clob);
        
    }

    @Override
    public Object getNullableResult(CallableStatement cs, int columnIndex) throws SQLException
    {
        Clob clob = cs.getClob(columnIndex);
        return _transformClobToString(clob);
    }
    
    /**
     * Transform CLOB value to String value.
     * @param clob the clob
     * @return the CLOB transformed to String.
     * @throws SQLException if an error occurred
     */
    protected String _transformClobToString(Clob clob) throws SQLException
    {
        try
        {
            String strValue = IOUtils.toString(clob.getCharacterStream());
            return CharMatcher.javaIsoControl().and(CharMatcher.anyOf("\r\n\t").negate()).removeFrom(strValue);
        }
        catch (IOException e)
        {
            __logger.warn("Can't transform clob to string", e);
            return null;
        }
        finally
        {
            clob.free();
        }
    }
}
