/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentstree;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for contents tree
 *
 */
public class ContentsTreeHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = ContentsTreeHelper.class.getName();
    
    /** The ametys object resolver instance */
    protected AmetysObjectResolver _ametysResolver;
    /** The tree configuration EP instance */
    protected TreeExtensionPoint _treeExtensionPoint;
    /** The content type EP instance */
    protected ContentTypeExtensionPoint _contentTypesEP;
    /** The content types helper instance */
    protected ContentTypesHelper _contentTypesHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _treeExtensionPoint = (TreeExtensionPoint) smanager.lookup(TreeExtensionPoint.ROLE);
        _contentTypesEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
    }
    
    /**
     * Determines if the content has children contents according the tree configuration
     * @param content the root content
     * @param treeConfiguration the tree configuration
     * @return true if the content has children contents
     */
    public boolean hasChildrenContent(Content content, TreeConfiguration treeConfiguration)
    {
        return !getChildrenContent(content, treeConfiguration).isEmpty();
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param parentContent the root content
     * @param treeConfiguration the tree configuration
     * @return the children content for each child attributes
     */
    public Map<String, List<Content>> getChildrenContent(Content parentContent, TreeConfiguration treeConfiguration)
    {
        Map<String, List<Content>> childrenContent = new HashMap<>();
        
        // For each content type of the content
        for (String contentTypeId : parentContent.getTypes())
        {
            // Loop over all possible elements for the tree
            for (TreeConfigurationElements treeConfigurationElements : treeConfiguration.getElements())
            {
                // Check for a match between the element and the content type of the content
                for (TreeConfigurationContentType treeConfigurationContentType : treeConfigurationElements.getContentTypesConfiguration())
                {
                    if (treeConfigurationContentType.getContentTypesIds().contains(contentTypeId))
                    {
                        ContentType contentType = _contentTypesEP.getExtension(contentTypeId);
                        
                        // Add all required children for this element
                        for (TreeConfigurationElementsChild treeConfigurationElementsChild : treeConfigurationElements.getChildren())
                        {
                            if (treeConfigurationElementsChild instanceof AttributeTreeConfigurationElementsChild)
                            {
                                // Get the attribute
                                Map<String, List<Content>> contents = _handleAttributeTreeConfigurationElementsChild(contentType, parentContent, (AttributeTreeConfigurationElementsChild) treeConfigurationElementsChild, treeConfiguration);
                                _merge(childrenContent, contents);
                            }
                            else
                            {
                                throw new IllegalArgumentException("The child configuration element class <" + treeConfigurationElementsChild + "> is not supported in tree '" + treeConfiguration.getId() + "'");
                            }
                        }
                    }
                }
                
            }
        }
        
        return childrenContent;
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param contentId the parent content
     * @param treeId the tree configuration
     * @return the children content
     */
    @Callable
    public Map<String, Object> getChildrenContent(String contentId, String treeId)
    {
        TreeConfiguration treeConfiguration = _getTreeConfiguration(treeId);
        Content parentContent = _getParentContent(contentId);
        
        Map<String, Object> infos = new HashMap<>();

        _addChildren(parentContent, treeConfiguration, infos);
        
        infos.putAll(getNodeInformations(contentId));

        return infos;
    }
    
    /**
     * Add the json info to list the children of a content
     * @param content The content
     * @param treeConfiguration The current tree configuration
     * @param infos The infos where to add the children key
     */
    protected void _addChildren(Content content, TreeConfiguration treeConfiguration, Map<String, Object> infos)
    {
        Map<String, List<Content>> children = getChildrenContent(content, treeConfiguration);
        
        boolean hasAtLeastOneAutoExpand = hasAutoExpandTargets(treeConfiguration);

        List<Map<String, Object>> childrenInfos = new ArrayList<>();
        infos.put("children", childrenInfos);

        for (String attributePath : children.keySet())
        {
            for (Content childContent : children.get(attributePath))
            {
                boolean expand = hasAtLeastOneAutoExpand && !isAnAutoExpandTarget(treeConfiguration, childContent);
                
                Map<String, Object> childInfo = content2Json(childContent);
                childInfo.put("metadataPath", attributePath);
                childInfo.put("expanded", expand);
                
                if (expand)
                {
                    _addChildren(childContent, treeConfiguration, childInfo);
                }
                
                if (!hasChildrenContent(childContent, treeConfiguration))
                {
                    // childInfo.put("leaf", true);
                    childInfo.put("children", Collections.EMPTY_LIST);
                }
                else
                {
                    childInfo.put("leaf", false);
                    childInfo.put("isExpanded", false);
                }

                childrenInfos.add(childInfo);
            }
        }

    }
    
    /**
     * Get the path of children content which match filter regexp
     * @param parentContentId The id of content to start search
     * @param treeId The id of tree configuration
     * @param value the value to match
     * @return the matching paths composed by contents id separated by ';'
     */
    @Callable
    public List<String> filterChildrenContentByRegExp(String parentContentId, String treeId, String value)
    {
        List<String> matchingPaths = new ArrayList<>();

        Content parentContent = _ametysResolver.resolveById(parentContentId);
        TreeConfiguration treeConfiguration = _treeExtensionPoint.getExtension(treeId);
        
        String toMatch = StringUtils.stripAccents(value.toLowerCase()).trim();
        
        Map<String, List<Content>> childrenContentByAttributes = getChildrenContent(parentContent, treeConfiguration);
        for (List<Content> childrenContent : childrenContentByAttributes.values())
        {
            for (Content childContent : childrenContent)
            {
                _getMatchingContents(childContent, toMatch, treeConfiguration, matchingPaths, parentContentId);
            }
        }
        
        return matchingPaths;
    }
    
    private void _getMatchingContents(Content content, String value, TreeConfiguration treeConfiguration, List<String> matchingPaths, String parentPath)
    {
        if (isContentMatching(content, value))
        {
            matchingPaths.add(parentPath + ";" + content.getId());
        }
        
        Map<String, List<Content>> childrenContentByAttributes = getChildrenContent(content, treeConfiguration);
        for (List<Content> childrenContent : childrenContentByAttributes.values())
        {
            for (Content childContent : childrenContent)
            {
                _getMatchingContents(childContent, value, treeConfiguration, matchingPaths, parentPath + ";" + content.getId());
            }
        }
    }
    
    /**
     * Determines if content matches the filter regexp
     * @param content the content
     * @param value the value to match
     * @return true if the content match
     */
    protected boolean isContentMatching(Content content, String value)
    {
        String title =  StringUtils.stripAccents(content.getTitle().toLowerCase());
        return title.contains(value);
    }
    
    /**
     * Get the root node informations
     * @param contentId The content
     * @param treeId The contents tree id
     * @return The informations
     */
    @Callable
    public Map<String, Object> getRootNodeInformations(String contentId, String treeId)
    {
        TreeConfiguration treeConfiguration = _getTreeConfiguration(treeId);
        Content content = _ametysResolver.resolveById(contentId);
        
        Map<String, Object> nodeInformations =  content2Json(content);
        nodeInformations.put("id", "root"); // Root id should not be random, for delete op
        _addChildren(content, treeConfiguration, nodeInformations); // auto expand first level + recursively if necessary

        return nodeInformations;
    }
    
    /**
     * Get the node informations
     * @param contentId The content
     * @return The informations
     */
    @Callable
    public Map<String, Object> getNodeInformations(String contentId)
    {
        Content content = _ametysResolver.resolveById(contentId);
        return content2Json(content);
    }
    
    /**
     * Get the default JSON representation of a content of the tree
     * @param content the content
     * @return the content as JSON
     */
    protected Map<String, Object> content2Json(Content content)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", "random-id-" + org.ametys.core.util.StringUtils.generateKey() + "-" + Math.round(Math.random() * 10_000));
        
        infos.put("contentId", content.getId());
        infos.put("contenttypesIds", content.getTypes());
        infos.put("name", content.getName());
        infos.put("title", content.getTitle());

        infos.put("iconGlyph", _contentTypesHelper.getIconGlyph(content));
        infos.put("iconDecorator", _contentTypesHelper.getIconDecorator(content));
        infos.put("iconSmall", _contentTypesHelper.getSmallIcon(content));
        infos.put("iconMedium", _contentTypesHelper.getMediumIcon(content));
        infos.put("iconLarge", _contentTypesHelper.getLargeIcon(content));
        
        return infos;
    }
    
    /**
     * Get the default JSON representation of a child content
     * @param content the content
     * @param attributePath the path of attribute holding this content
     * @return the content as JSON
     */
    public Map<String, Object> childContent2Json(Content content, String attributePath)
    {
        Map<String, Object> childInfo = content2Json(content);
        childInfo.put("metadataPath", attributePath);
        return childInfo;
    }
    
    private void _merge(Map<String, List<Content>> childrenContent, Map<String, List<Content>> contents)
    {
        for (String key : contents.keySet())
        {
            if (!childrenContent.containsKey(key))
            {
                childrenContent.put(key, new ArrayList<>());
            }
            
            List<Content> contentsList = childrenContent.get(key);
            contentsList.addAll(contents.get(key));
        }
    }

    private Map<String, List<Content>> _handleAttributeTreeConfigurationElementsChild(ContentType contentType, ModelAwareDataHolder dataHolder, AttributeTreeConfigurationElementsChild attributeTreeConfigurationElementsChild, TreeConfiguration treeConfiguration)
    {
        Map<String, List<Content>> childrenContent = new HashMap<>();
        
        String attributePath = attributeTreeConfigurationElementsChild.getPath();

        try
        {
            Map<String, List<Content>> contents = _handleAttribute(contentType, dataHolder, attributePath);
            _merge(childrenContent, contents);
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("An error occured on the tree configuration '" + treeConfiguration.getId() + "' getting for metadata '" + attributePath + "' on content type '" + contentType.getId() + "'", e);
        }

        return childrenContent;
    }

    private Map<String, List<Content>> _handleAttribute(ModelItemContainer modelItemContainer, ModelAwareDataHolder dataHolder, String attributePath)
    {
        Map<String, List<Content>> childrenContent = new HashMap<>();

        String currentModelItemName = StringUtils.substringBefore(attributePath, ModelItem.ITEM_PATH_SEPARATOR);
        
        ModelItem currentModelItem = modelItemContainer.getChild(currentModelItemName);
        if (currentModelItem == null)
        {
            throw new IllegalArgumentException("No attribute definition for " + currentModelItemName);
        }
        
        
        if (dataHolder.hasValue(currentModelItemName))
        {
            if (currentModelItem instanceof RepeaterDefinition)
            {
                ModelAwareRepeater repeater = dataHolder.getRepeater(currentModelItemName);
                for (ModelAwareRepeaterEntry entry : repeater.getEntries())
                {
                    String subMetadataId = StringUtils.substringAfter(attributePath, ModelItem.ITEM_PATH_SEPARATOR);
                    Map<String, List<Content>> contents = _handleAttribute((RepeaterDefinition) currentModelItem, entry, subMetadataId);

                    _merge(childrenContent, contents);
                }
            }
            else if (currentModelItem instanceof CompositeDefinition)
            {
                ModelAwareComposite metadata = dataHolder.getComposite(currentModelItemName);
                
                String subMetadataId = StringUtils.substringAfter(attributePath, ModelItem.ITEM_PATH_SEPARATOR);
                Map<String, List<Content>> contents = _handleAttribute((CompositeDefinition) currentModelItem, metadata, subMetadataId);

                _merge(childrenContent, contents);
            }
            else if (ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(currentModelItem.getType().getId()))
            {
                ContentValue[] contentValues = dataHolder.getValue(currentModelItemName);
                for (ContentValue contentValue : contentValues)
                {
                    String key = currentModelItem.getPath();
                    Optional<ModifiableContent> optContent = contentValue.getContentIfExists();
                    if (optContent.isPresent())
                    {
                        childrenContent.computeIfAbsent(key, k -> new ArrayList<>()).add(optContent.get());
                    }
                    else
                    {
                        getLogger().warn("On a data holder, the attribute '{}' is referencing a unexisting content: '{}'", key, contentValue.getContentId());
                    }
                }
            }
            else
            {
                throw new IllegalArgumentException("The metadata definition for " + currentModelItem.getPath() + " is not a content");
            }
        }
        
        return childrenContent;
    }
    
    /**
     * Get the tree configuration
     * @param treeId the tree id
     * @return the tree configuration
     */
    protected TreeConfiguration _getTreeConfiguration(String treeId)
    {
        if (StringUtils.isBlank(treeId))
        {
            throw new IllegalArgumentException("The tree information cannot be obtain, because 'tree' is null");
        }

        TreeConfiguration treeConfiguration = _treeExtensionPoint.getExtension(treeId);
        if (treeConfiguration == null)
        {
            throw new IllegalArgumentException("There is no tree configuration for '" + treeId + "'");
        }
        return treeConfiguration;
    }

    /**
     * Get the parent content of a tree
     * @param parentId the parent id
     * @return the parent content of a tree
     * @throws IllegalArgumentException if an exception occurred
     */
    protected Content _getParentContent(String parentId) throws IllegalArgumentException
    {
        if (StringUtils.isBlank(parentId))
        {
            throw new IllegalArgumentException("The tree information cannot be obtain, because 'node' is null");
        }
        
        try
        {
            return _ametysResolver.resolveById(parentId);
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("The tree configuration cannot be used on an object that is not a content: " + parentId, e);
        }

    }

    /**
     * Should auto expand until some kind of node?
     * @param treeConfiguration The tree configuration
     * @return true if should auto expand
     */
    protected boolean hasAutoExpandTargets(TreeConfiguration treeConfiguration)
    {
        return treeConfiguration.getElements()
            .stream()
            .map(TreeConfigurationElements::getContentTypesConfiguration)
            .flatMap(Collection::stream)
            .anyMatch(TreeConfigurationContentType::autoExpandTarget);
    }
    
    /**
     * Should auto expand until some kind of node?
     * @param treeConfiguration The tree configuration
     * @param content The content involved
     * @return true if should auto expand to it
     */
    protected boolean isAnAutoExpandTarget(TreeConfiguration treeConfiguration, Content content)
    {
        List<String> contentTypes = Arrays.asList(content.getTypes());
        
        return treeConfiguration.getElements()
                .stream()
                .map(TreeConfigurationElements::getContentTypesConfiguration)
                .flatMap(Collection::stream)
                .filter(TreeConfigurationContentType::autoExpandTarget)
                .map(TreeConfigurationContentType::getContentTypesIds)
                .anyMatch(ct -> { Set<String> hs = new HashSet<>(ct); hs.retainAll(contentTypes); return hs.size() > 0; });
    }
}
