/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentstree.ui;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.contentstree.TreeConfiguration;
import org.ametys.plugins.contentstree.TreeConfigurationContentType;
import org.ametys.plugins.contentstree.TreeConfigurationElements;
import org.ametys.plugins.contentstree.TreeExtensionPoint;
import org.ametys.core.ui.StaticClientSideElement;

/**
 * This class act as a StaticClientSideElement but do handle a new tag "tree-config" that lead to a file containing the tree configuration
 */
public class OpenTreeControllerClientSideElement extends StaticClientSideElement
{
    /** The tree configuration EP */
    protected TreeExtensionPoint _treeExtensionPoint;

    /** During startup will contains the configured tree config id. Null once initialized */
    protected String _treeConfigId;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _treeExtensionPoint = (TreeExtensionPoint) smanager.lookup(TreeExtensionPoint.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _treeConfigId = configuration.getChild("tree-config").getValue();
    }
    
    /**
     * Lazy configuration of the component
     */
    protected void _lazyConfigure()
    {
        if (_treeConfigId != null)
        {
            TreeConfiguration treeConfiguration = _treeExtensionPoint.getExtension(_treeConfigId);
            
            _script.getParameters().put("opentool-id", treeConfiguration.getUIToolRole());
            
            Map<String, Object> openToolParams = new HashMap<>();
            openToolParams.put("treeId", _treeConfigId);
            _script.getParameters().put("opentool-params", openToolParams); 

            
            Set<String> busMessageTypesRegExp = new HashSet<>();
            Set<String> contentTypeIdsRegExp = new HashSet<>();
            
            for (TreeConfigurationElements element : treeConfiguration.getElements())
            {
                for (TreeConfigurationContentType contentTypeInfo : element.getContentTypesConfiguration())
                {
                    if (contentTypeInfo.canBeRoot())
                    {
                        busMessageTypesRegExp.add("^" + contentTypeInfo.getMessageBusType() + "$");
                        
                        for (String id : contentTypeInfo.getContentTypesIds())
                        {
                            contentTypeIdsRegExp.add("^" + id + "$");
                        }
                    }
                }
            }
            
            _script.getParameters().put("selection-target-id", StringUtils.join(busMessageTypesRegExp, "|"));
            
            Map<String, Object> selectionTargetParameter = new HashMap<>();
            selectionTargetParameter.put("name", "^types$");
            selectionTargetParameter.put("value", StringUtils.join(contentTypeIdsRegExp, "|"));
            
            _script.getParameters().put("selection-target-parameter", selectionTargetParameter);

            _treeConfigId = null;
        }
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        _lazyConfigure();
        
        return super.getScripts(ignoreRights, contextParameters);
    }
}
