/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentstree;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Static implementation based on configuration. See doc for template.
 */
public class StaticTreeConfiguration implements TreeConfiguration, Configurable, Serviceable, PluginAware
{
    /** The ui tool role */
    protected String _uiToolRole;
    /** The collection of elements */
    protected Set<TreeConfigurationElements> _elements;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypesEP;
    /** The excalibur source resolver */
    protected SourceResolver _sourceResolver;
    /** The plugin name */
    protected String _pluginName;
    /** The extension id */
    protected String _id;
    
    private Configuration _configuration;

    @Override
    public String getId()
    {
        return _id;
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public void configure(Configuration initialConfiguration) throws ConfigurationException
    {
        _id = initialConfiguration.getAttribute("id");
        
        _configuration = _getDeportedConfiguration(initialConfiguration);
    }
    
    private void _lazyConfigure()
    {
        if (_configuration != null)
        {
            try
            {
                _uiToolRole = _configuration.getAttribute("id");
                _elements = new HashSet<>();
                
                for (Configuration contentTypeConfiguration : _configuration.getChild("elements").getChildren("content-type"))
                {
                    Collection<TreeConfigurationContentType> contentTypesInfo = getContentTypesInfos(contentTypeConfiguration.getChild("ids"));
                    Collection<TreeConfigurationElementsChild> childNodes = getChildNodes(contentTypeConfiguration.getChild("children"));
                    
                    TreeConfigurationElements element = new TreeConfigurationElements(contentTypesInfo, childNodes);
                    _elements.add(element);
                }
                
                _configuration = null;
            }
            catch (ConfigurationException e)
            {
                throw new IllegalStateException("An error occured while configuring tree '" + _id + "'", e);
            }
        }
    }

    private Configuration _getDeportedConfiguration(Configuration initialConfiguration) throws ConfigurationException
    {
        String treeConfigURL = initialConfiguration.getChild("tree-config").getValue();
        Source treeConfigSource = null;
        try
        {
            treeConfigSource = _sourceResolver.resolveURI(treeConfigURL, "plugin:" + _pluginName + "://", null);
            
            try (InputStream is = treeConfigSource.getInputStream())
            {
                return new DefaultConfigurationBuilder().build(is);
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Cannot open the file '" + treeConfigURL + "'", initialConfiguration, e);
        }
        finally
        {
            _sourceResolver.release(treeConfigSource);
        }
    }

    /**
     * Get the child nodes
     * @param configuration The configuration
     * @return the child nodes
     * @throws ConfigurationException if configuration is invalid
     */
    protected Collection<TreeConfigurationElementsChild> getChildNodes(Configuration configuration) throws ConfigurationException
    {
        Collection<TreeConfigurationElementsChild> childnodes = new ArrayList<>();
        
        for (Configuration childConfiguration : configuration.getChildren())
        {
            if (StringUtils.equals(childConfiguration.getName(), "metadata"))
            {
                String id = childConfiguration.getAttribute("id");
                
                TreeConfigurationElementsChild child = new AttributeTreeConfigurationElementsChild(id);
                childnodes.add(child);
            }
            else
            {
                throw new ConfigurationException("Unknown child of type '" + childConfiguration.getName() + "'", childConfiguration);
            }
        }
        
        return childnodes;
    }

    /**
     * Get the contents infos using a configuration
     * @param configuration the configuration
     * @return the contents infos using a configuration
     * @throws ConfigurationException if an error occurred
     */
    protected Collection<TreeConfigurationContentType> getContentTypesInfos(Configuration configuration) throws ConfigurationException
    {
        Collection<TreeConfigurationContentType> contentTypesInfo = new ArrayList<>();

        boolean elementsCanBeRoot = configuration.getAttributeAsBoolean("can-be-root", false);
        boolean elementsAutoExpandTarget = configuration.getAttributeAsBoolean("auto-expand-target", false);
        boolean elementsInherit = configuration.getAttributeAsBoolean("inherit", false);
        String elementsBusMessageType = configuration.getAttribute("bus-message-type", "content");
        
        for (Configuration idConfiguration : configuration.getChildren("id"))
        {
            TreeConfigurationContentType contentTypeInfo = getContentTypeInfo(idConfiguration, elementsCanBeRoot, elementsAutoExpandTarget, elementsInherit, elementsBusMessageType);
            contentTypesInfo.add(contentTypeInfo);
        }
        
        return contentTypesInfo;
    }
    
    /**
     * Get the content info using a configuration
     * @param configuration The configuration
     * @param defaultCanBeRoot default value for root property
     * @param defaultAutoExpandToIt default value for autoExpandToIt property
     * @param defaultInherit default value for inherit property
     * @param defaultBusMessageType default type of message
     * @return the content info using a configuration
     * @throws ConfigurationException if the configuration is invalid
     */
    protected TreeConfigurationContentType getContentTypeInfo(Configuration configuration, boolean defaultCanBeRoot, boolean defaultAutoExpandToIt, boolean defaultInherit, String defaultBusMessageType) throws ConfigurationException
    {
        boolean canBeRoot = configuration.getAttributeAsBoolean("can-be-root", defaultCanBeRoot);
        boolean autoExpandToIt = configuration.getAttributeAsBoolean("auto-expand-to-it", defaultAutoExpandToIt);
        String mainContentTypeId = configuration.getValue();
        
        Collection<String> typesIds = new ArrayList<>();
        typesIds.add(mainContentTypeId);
        
        boolean inherit = configuration.getAttributeAsBoolean("inherit", defaultInherit);
        if (inherit)
        {
            typesIds.addAll(_contentTypesEP.getSubTypes(mainContentTypeId));
        }
        
        String busMessageType = configuration.getAttribute("bus-message-type", defaultBusMessageType);
        
        return new TreeConfigurationContentType(canBeRoot, autoExpandToIt, typesIds, busMessageType);
    }
    
    @Override
    public String getUIToolRole()
    {
        _lazyConfigure();
        return _uiToolRole;
    }

    @Override
    public Set<TreeConfigurationElements> getElements()
    {
        _lazyConfigure();
        return _elements;
    }
}
