/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentstree.ui;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.contentstree.AttributeTreeConfigurationElementsChild;
import org.ametys.plugins.contentstree.ContentsTreeHelper;
import org.ametys.plugins.contentstree.TreeConfiguration;
import org.ametys.plugins.contentstree.TreeConfigurationContentType;
import org.ametys.plugins.contentstree.TreeConfigurationElements;
import org.ametys.plugins.contentstree.TreeConfigurationElementsChild;
import org.ametys.plugins.contentstree.TreeExtensionPoint;
import org.ametys.runtime.model.ModelItem;

/**
 * This client side element automatically add elements to the js configuration in relation with the &lt;tree-config&gt;
 */
public class TreeToolClientSideElement extends StaticClientSideElement
{
    /** The content types helper instance */
    protected ContentsTreeHelper _treeHelper;
    /** The tree configuration EP */
    protected TreeExtensionPoint _treeExtensionPoint;
    /** During startup will contains the configured tree config id. Null once initialized */
    protected String _treeConfigId;
    /** The content type helper */
    protected ContentTypesHelper _contentTypesHelper;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _treeHelper = (ContentsTreeHelper) smanager.lookup(ContentsTreeHelper.ROLE);
        _treeExtensionPoint = (TreeExtensionPoint) smanager.lookup(TreeExtensionPoint.ROLE);
        _contentTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _treeConfigId = configuration.getChild("tree-config").getValue();
    }
    
    /**
     * Lazy configuration of the component
     */
    protected void _lazyConfigure()
    {
        if (_treeConfigId != null)
        {
            TreeConfiguration treeConfiguration = _treeExtensionPoint.getExtension(_treeConfigId);

            _script.getParameters().put("treeId", _treeConfigId);
            
            Set<String> busMessageTypesRegExp = new HashSet<>();
            Set<String> contentTypeIdsRegExp = new HashSet<>();
            Map<String, String> contenttypeAndMessagebustype = new HashMap<>();
            Map<String, Map<String, List<String>>> contentTypeAndAttributePathsByContentType = new HashMap<>();
            
            for (TreeConfigurationElements element : treeConfiguration.getElements())
            {
                for (TreeConfigurationContentType contentTypeInfo : element.getContentTypesConfiguration())
                {
                    if (contentTypeInfo.canBeRoot())
                    {
                        busMessageTypesRegExp.add("^" + contentTypeInfo.getMessageBusType() + "$"); 
                    }
                    
                    for (String id : contentTypeInfo.getContentTypesIds())
                    {
                        Map<String, List<String>> contentTypeAndAttributePaths = new HashMap<>();
                        contentTypeAndAttributePathsByContentType.put(id, contentTypeAndAttributePaths);
                        
                        if (contentTypeInfo.canBeRoot())
                        {
                            contentTypeIdsRegExp.add("^" + id + "$");
                        }
                        
                        contenttypeAndMessagebustype.put(id, contentTypeInfo.getMessageBusType());
                        
                        for (TreeConfigurationElementsChild treeConfigurationElementsChild : element.getChildren())
                        {
                            if (treeConfigurationElementsChild instanceof AttributeTreeConfigurationElementsChild)
                            {
                                AttributeTreeConfigurationElementsChild attributeTreeConfigurationElementsChild = (AttributeTreeConfigurationElementsChild) treeConfigurationElementsChild;
                                String attributePath = attributeTreeConfigurationElementsChild.getPath();
                                
                                List<String> contentTypes = new ArrayList<>();
                                contentTypeAndAttributePaths.put(attributePath, contentTypes);
                                
                                ContentType contentType = _contentTypeExtensionPoint.getExtension(id);
                                ModelItem metadataDefinition = contentType.getModelItem(attributePath);
                                if (metadataDefinition instanceof ContentAttributeDefinition)
                                {
                                    String targetContentTypeId = ((ContentAttributeDefinition) metadataDefinition).getContentTypeId();
                                    contentTypes.add(targetContentTypeId);
                                    contentTypes.addAll(_contentTypeExtensionPoint.getSubTypes(targetContentTypeId));
                                }
                            }
                        }
                    }
                }
            }
            _script.getParameters().put("messagebustype-by-contenttype", contenttypeAndMessagebustype);
            
            _script.getParameters().put("metadatapaths-by-contenttype", contentTypeAndAttributePathsByContentType);
            
            _script.getParameters().put("selection-target-id", StringUtils.join(busMessageTypesRegExp, "|"));
            
            Map<String, Object> selectionTargetParameter = new HashMap<>();
            selectionTargetParameter.put("name", "^types$");
            selectionTargetParameter.put("value", StringUtils.join(contentTypeIdsRegExp, "|"));
            
            _script.getParameters().put("selection-target-parameter", selectionTargetParameter);
        }
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        _lazyConfigure();
        
        return super.getScripts(ignoreRights, contextParameters);
    }
    
    /**
     * Get the root node informations
     * @param contentId The content
     * @param treeId The tree config id
     * @return The informations
     * @throws Exception if an error occurred
     */
    @Callable
    public Map<String, Object> getRootNodeInformations(String contentId, String treeId) throws Exception
    {
        return _treeHelper.getRootNodeInformations(contentId, treeId);
    }
}
