/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.Binary;
import org.ametys.cms.data.Geocode;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.model.ViewHelper;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.sitemap.Sitemap;

import com.opensymphony.workflow.WorkflowException;

/**
 * Helper for creating generic contents  
 */
public class ContentCreationHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = ContentCreationHelper.class.getName();
    
    private static final String _FILLING_DATA_FOLDER_URI = "plugin:data-filler://filling-data/resources/";

    private static final int _INITIAL_ACTION_ID = 1;

    private static final int _EDIT_ACTION_ID = 2;

    private static final int _VALIDATE_ACTION_ID = 4;

    private static final String _CONTENT_WORKFLOW_NAME = "content";

    /** The i18nUtils */
    private I18nUtils _i18nUtils;
    /** The ametys object resolver */
    private AmetysObjectResolver _resolver;
    /** The source resolver */
    private SourceResolver _sourceResolver;
    /** the current user provider*/
    private CurrentUserProvider _currentUserProvider;
    /** The content workflow helper. */
    private ContentWorkflowHelper _workflowHelper;
    /** The workflow provider. */
    private WorkflowProvider _workflowProvider;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _workflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }
    
    /**
     * Generate generic values for the view and return them in a map
     * @param viewItemContainer the view to fill
     * @return the Map of Object containing the values
     */
    @SuppressWarnings("unchecked")
    public  Map<String, Object> getGenericValuesForView(ViewItemContainer viewItemContainer)
    {
        Map<String, Object> result = new HashMap<>();
        ViewHelper.visitView(viewItemContainer,
            (element, definition) -> {
                // simple element
                String name = definition.getName();
                
                Object value;
                value = definition.getDefaultValue();
                // Use default value if it exist
                if (value != null)
                {
                    result.put(name, value);
                }
                else
                {
                    // We don't fill the dublin core value
                    // We also ignore title as we already set it and don't want a generic value instead
                    if (!name.contains("dc") && !name.contains("title"))
                    {
                        try
                        {
                            value = _getGenericValue(definition);
                            if (value != null)
                            {
                                if (definition.isMultiple())
                                {
                                    result.put(name, Set.of(value));
                                }
                                else
                                {
                                    result.put(name, value);
                                }
                            }
                        }
                        catch (IOException e)
                        {
                            throw new RuntimeException(e);
                        }
                    }
                }
            },
            (group, definition) -> {
                // composite
                String name = definition.getName();
                Map<String, Object> value = getGenericValuesForView(group);
                result.put(name, value);
            },
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                Map<String, Object> value = getGenericValuesForView(group);
                int size = (definition.getInitialSize() >= 1) ? definition.getInitialSize()
                           : (definition.getMinSize() >= 1) ? definition.getMinSize()
                           : (definition.getMaxSize() >= 1) ? definition.getMaxSize() : 2;

                result.put(name, Collections.nCopies(size, value));
            },
            group -> result.putAll(getGenericValuesForView(group))
        );
        return result;
    }

    /** Provide a generic value based on {@code definition} type.
     * @param definition the ElementDefinition we want to provide a value to
     * @return a value compatible with the element definition
     * @throws IOException if the type required a value from a source that failed to be read
     */
    private <T> Object _getGenericValue(ElementDefinition<T> definition) throws IOException
    {
        switch (definition.getType().getId())
        {
            // FIXME DATAFILLER-16 Attribute of type user and multilingual string are not handle in generic content
            // case ModelItemTypeConstants.USER_ELEMENT_TYPE_ID:
            // case ModelItemTypeConstants.MULTILINGUAL_STRING_ELEMENT_TYPE_ID:
            case org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID:
                return _getGenericString(definition);
                
            case ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID:
                return null;
                
            case org.ametys.runtime.model.type.ModelItemTypeConstants.LONG_TYPE_ID:
                return _getGenericLong(definition);
                
            case org.ametys.runtime.model.type.ModelItemTypeConstants.BOOLEAN_TYPE_ID:
                return true;
                
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATE_TYPE_ID:
                return LocalDate.now();
                
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATETIME_TYPE_ID:
                return ZonedDateTime.now();
                
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DOUBLE_TYPE_ID:
                return 1.3;
                
            case ModelItemTypeConstants.GEOCODE_ELEMENT_TYPE_ID:
                return new Geocode(50.6637, -4.7593);
                
            case ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID:
            case ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID:
                return _getGenericFile(definition);
                
            case ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID:
                return _getGenericRichText();
                
            case ModelItemTypeConstants.USER_ELEMENT_TYPE_ID:
                return _getGenericUser();
            default:
                getLogger().warn("Unsupported type '{}'. The attribute will be ignored.", definition.getType());
                return null;
        }
    }

    private String _getGenericString(ElementDefinition definition)
    {
        // relying on attribute name is dangerous as this is not standardized.
        // But we don't have better choice to be handle validator
        switch (definition.getName())
        {
            case "tags" :
                return null;
                
            case "email" :
            case "mail" :
                return "contact@ametys.org";
                
            case "link" :
            case "url" :
            case "website" :
                return "https://ametys.org";
                
            default:
                return "Lorem ipsum et tutti quanti";
        }
    }

    private Long _getGenericLong(ElementDefinition definition)
    {
        switch (definition.getName())
        {
            case "height" :
            case "width" :
                return null;
            default :
                return 4L;
        }
    }

    private <T> Object _getGenericFile(ElementDefinition<T> definition) throws MalformedURLException, IOException, SourceNotFoundException
    {
        String filter = Optional.of(definition).map(ElementDefinition::getWidgetParameters)
                                               .map(params -> params.get("filter"))
                                               .map(I18nizableText::getLabel)
                                               .orElse(StringUtils.EMPTY);
        Source ressourceFile = null;
        String filename;
        String mimeType;
        try
        {
            switch (filter)
            {
                case "audio":
                    mimeType = "audio/mp3";
                    filename = "sound.mp3";
                    break;
                case "image":
                    mimeType = "image/jpeg";
                    filename = "logo.jpg";
                    break;
                case "video":
                    mimeType = "video/mp4";
                    filename = "video.mp4";
                    break;
                case "pdf":
                    mimeType = "application/pdf";
                    filename = "document.pdf";
                    break;
                default:
                    // Handle definition without filter but with allowed extension instead
                    String allowedExtension = Optional.of(definition).map(ElementDefinition::getWidgetParameters)
                                                                     .map(params -> params.get("allowExtensions"))
                                                                     .map(I18nizableText::getLabel)
                                                                     .orElse(StringUtils.EMPTY);
                    // if there is no restriction, we provide a picture
                    if (StringUtils.isBlank(allowedExtension))
                    {
                        mimeType = "image/jpeg";
                        filename = "logo.jpg";
                    }
                    else if (allowedExtension.contains("mp3"))
                    {
                        mimeType = "audio/mp3";
                        filename = "sound.mp3";
                    }
                    else if (allowedExtension.contains("mp4"))
                    {
                        mimeType = "video/mp4";
                        filename = "video.mp4";
                    }
                    else if (allowedExtension.contains("pdf"))
                    {
                        mimeType = "application/pdf";
                        filename = "document.pdf";
                    }
                    else if (allowedExtension.contains("vtt"))
                    {
                        mimeType = "text/vtt";
                        filename = "subtitle.vtt";
                    }
                    else
                    {
                        getLogger().info(allowedExtension + " is not supported. The file attribute will be ignored");
                        return null;
                    }
            }
            ressourceFile = _sourceResolver.resolveURI(_FILLING_DATA_FOLDER_URI + filename);
            Binary file = new Binary();
            try (InputStream is = ressourceFile.getInputStream();)
            {
                file.setInputStream(is);
                file.setMimeType(mimeType);
                file.setFilename(filename);
            }
            return file;
        }
        finally
        {
            _sourceResolver.release(ressourceFile);
        }
    }

    private RichText _getGenericRichText() throws MalformedURLException, IOException, SourceNotFoundException
    {
        RichText text = new RichText();
        text.setEncoding("UTF-8");
        text.setLastModificationDate(ZonedDateTime.now());
        text.setMimeType("text/xml");
        Source docBookSource = null;
        InputStream is = null;
        try
        {
            docBookSource = _sourceResolver.resolveURI(_FILLING_DATA_FOLDER_URI + "richtext.xml");
            is = docBookSource.getInputStream();
            text.setInputStream(is);
        }
        finally
        {
            IOUtils.closeQuietly(is);
            _sourceResolver.release(docBookSource);
        }
        return text;
    }
    
    private UserIdentity _getGenericUser()
    {
        return _currentUserProvider.getUser();
    }

    /**
     * Get content by its name
     * @param contentName the content's name
     * @param sitemap the current sitemap
     * @return the content if exist
     */
    public WorkflowAwareContent getContent(String contentName, Sitemap sitemap)
    {
        // get the content if the content was already created
        String siteName = sitemap.getSiteName();
        String language = sitemap.getSitemapName();
        
        LanguageExpression langExp = new LanguageExpression(Operator.EQ, language);
        StringExpression siteExp = new StringExpression("site", Operator.EQ, siteName);
    
        String xPathQuery = QueryHelper.getXPathQuery(contentName, "ametys:content", new AndExpression(langExp, siteExp));
        AmetysObjectIterable<WorkflowAwareContent> contents = _resolver.query(xPathQuery);
        if (contents.getSize() != 0)
        {
            return contents.iterator().next();
        }
        return null;
    }

    /**
     * Create a workflow with an empty content of type {@code contentType}
     * @param contentType the type of content to create
     * @param sitemap the sitemap of the content
     * @param contentName the name that will be used to create the content
     * @return the empty {@code WorkflowAwareContent} initialized
     * @throws WorkflowException if an error occurred during the initialization
     */
    public WorkflowAwareContent initializeContent(ContentType contentType, Sitemap sitemap, String contentName) throws WorkflowException
    {
        String siteName = sitemap.getSiteName();
        String sitemapName = sitemap.getSitemapName();
        String contentTitle = _i18nUtils.translate(contentType.getLabel(), sitemapName);
    
        Map<String, Object> params = new HashMap<>();
    
        // Workflow result.
        Map<String, Object> workflowResult = new HashMap<>();
        params.put(AbstractWorkflowComponent.RESULT_MAP_KEY, workflowResult);
    
        // Workflow parameters.
        params.put(org.ametys.web.workflow.CreateContentFunction.SITE_KEY, siteName);
        params.put(CreateContentFunction.CONTENT_NAME_KEY, contentName);
        params.put(CreateContentFunction.CONTENT_TITLE_KEY, contentTitle);
        params.put(CreateContentFunction.CONTENT_TYPES_KEY, new String[] {contentType.getId()});
        params.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, sitemapName);
    
        // Trigger the creation.
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
        workflow.initialize(contentType.getDefaultWorkflowName().orElse(_CONTENT_WORKFLOW_NAME), _INITIAL_ACTION_ID, params);
    
        // Get the content in the results and return it.
        WorkflowAwareContent content = (WorkflowAwareContent) workflowResult.get(AbstractContentWorkflowComponent.CONTENT_KEY);
        return content;
    }

    /**
     * Edit content with map of values
     * @param content the content to edit
     * @param contentType the type of content
     * @param values the map of values to set
     * @return the content
     */
    public WorkflowAwareContent editContent(WorkflowAwareContent content, ContentType contentType, Map<String, Object> values)
    {
        // It is OK if the edition or validation of the content fail. We don't want it to be a blocker
        try
        {
            _workflowHelper.editContent(content, values, _EDIT_ACTION_ID);
            if (_workflowHelper.isAvailableAction(content, _VALIDATE_ACTION_ID))
            {
                _workflowHelper.doAction(content, _VALIDATE_ACTION_ID);
            }
            else
            {
                getLogger().warn("Can't validate content for type {}. The content is not in a valid state after being filled with generic data.", contentType.getId());
            }
        }
        catch (WorkflowException e)
        {
            getLogger().warn("Something went wrong when trying to edit or validate content of type {}.", contentType.getId(), e);
        }
        return content;
    }
}
