/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Stream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.AutoCloseInputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentDAO.TagMode;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentReturnable;
import org.ametys.web.repository.page.ContentTypesAssignmentHandler;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.ZoneDAO;
import org.ametys.web.repository.page.ZoneItemManager;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinTemplateZone;
import org.ametys.web.skin.SkinsManager;

import com.opensymphony.workflow.WorkflowException;

/**
 * Create and add generic content in test pages to test render, skins, and services
 */
public class GenericContentCreationManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = GenericContentCreationManager.class.getName();

    private static final String _FILE_DIRECTORY_CONTENTS_URI = "/WEB-INF/param/datafill/contents";

    private static final String _FILLING_DATA_FOLDER_URI = "plugin:data-filler://filling-data/resources/";

    /** The content type extension point */
    private ContentTypeExtensionPoint _cTypeEP;
    /** The content type assignment handler */
    private ContentTypesAssignmentHandler _cTypeHandler;
    /** The content DAO */
    private ContentDAO _contentDAO;
    /** The cocoon context */
    private org.apache.cocoon.environment.Context _context;
    /** The i18nUtils */
    private I18nUtils _i18nUtils;
    /** The page DAO */
    private PageDAO _pageDAO;
    /** The Sitemap creation Manager */
    private SitemapPopulator _sitemapPopulator;
    /** The source resolver */
    private SourceResolver _sourceResolver;
    /** The skin manager */
    private SkinsManager _skinsManager;
    /** The zone DAO */
    private ZoneDAO _zoneDAO;

    private ZoneItemManager _zoneItemManager;

    private ContentCreationHelper _contentCreationHelper;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = (org.apache.cocoon.environment.Context) context.get(org.apache.cocoon.Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentCreationHelper = (ContentCreationHelper) manager.lookup(ContentCreationHelper.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _cTypeHandler = (ContentTypesAssignmentHandler) manager.lookup(ContentTypesAssignmentHandler.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _sitemapPopulator = (SitemapPopulator) manager.lookup(SitemapPopulator.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _skinsManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _zoneDAO = (ZoneDAO) manager.lookup(ZoneDAO.ROLE);
        _zoneItemManager = (ZoneItemManager) manager.lookup(ZoneItemManager.ROLE);
    }
    

    /**
     * Create test pages filled with generic content for test purposes
     * @param sitemap the sitemap where the content will be inserted
     * @throws Exception when the page creation fails
     */
    public void createGenericContentPages(Sitemap sitemap) throws Exception
    {
        // create the content we want to insert
        Collection<Content> createdContents = _getOrCreateContentsForTestPages(sitemap);

        // then had each content to different test pages
        ModifiablePage autogeneratedSection = _sitemapPopulator.getOrCreatePage(sitemap, "Test des gabarits");
        for (Content content : createdContents)
        {
            _createPagesForContent(autogeneratedSection, content);
        }
    }

    /**
     * Create multiple contents for test purpose.
     * At least one content will be created for each {@link ContentType} available in the sitemap.
     * Content will either be defined according to definition provided by the user in the configuration files
     * or based on the ContentType.
     * @param sitemap the sitemap where the content will be inserted
     * @return a collection of all the content created
     */
    private Collection<Content> _getOrCreateContentsForTestPages(Sitemap sitemap)
    {
        Comparator<Content> comp = Comparator.<Content, String>comparing(Content::getTitle);
        Collection<Content> createdContents = new TreeSet<>(comp);
        createdContents.addAll(_getOrCreateSpecificContents(sitemap));
        
        // Get the list of contents defined in WEB_INF params
        Pair<Set<String>, Collection<Content>> result = _getOrCreateUserDefinedContents(sitemap);
        Set<String> overideContentTypeIds = result.getLeft();
        createdContents.addAll(result.getRight());
        
        Set<String> availableContentTypeIds = _cTypeEP.getExtensionsIds();
        for (String contentTypeID : availableContentTypeIds)
        {
            ContentType contentType = _cTypeEP.getExtension(contentTypeID);
            // We exclude contentType that should not be possible to attach to a page.
            // Some contentType might be missed. Mainly contentType with restriction to template.
            // If this is a problem, it can be improved by computing the list of allowed content by
            // every zone of every template and only including contentType in this list. As this is much more
            // expansive, we keep this logic for the moment.
            if (!contentType.isPrivate() && !contentType.isReferenceTable() && !contentType.isMixin() && !contentType.isAbstract())
            {
                // check if the contentType is overridden in WEB_INF
                if (!overideContentTypeIds.contains(contentTypeID))
                {
                    try
                    {
                        
                        Content content = _getOrCreateGenericContent(contentType, sitemap);
                        createdContents.add(content);
                        if (content.hasDefinition("illustration"))
                        {
                            Content copyContent = _getOrCreateGenericContentWithoutIllustration(contentType, sitemap);
                            if (copyContent != null)
                            {
                                createdContents.add(copyContent);
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        getLogger().warn("Failed to create content for contentType {}", contentTypeID, e);
                    }
                }
            }
        }
        return createdContents;
    }

    private Set<Content> _getOrCreateSpecificContents(Sitemap sitemap)
    {
        Set<Content> specificContent = new HashSet<>();
        try
        {
            specificContent.add(_getOrCreateDocbookDemonstratorContent(sitemap));
        }
        catch (WorkflowException e)
        {
            getLogger().warn("Failed to create Docbook demonstrator content");
        }
        
        try
        {
            specificContent.add(_getOrCreateFormDemonstratorContent(sitemap));
        }
        catch (WorkflowException e)
        {
            getLogger().warn("Failed to create rich text with form content");
        }
        return specificContent;
    }

    private Content _getOrCreateFormDemonstratorContent(Sitemap sitemap) throws WorkflowException
    {
        String title = "Richtext avec formulaire";
        String contentName = "richtext-form";
        
        WorkflowAwareContent content = _contentCreationHelper.getContent(contentName, sitemap);
        
        if (content == null)
        {
            RichText text = new RichText();
            text.setEncoding("UTF-8");
            text.setLastModificationDate(ZonedDateTime.now());
            text.setMimeType("text/xml");

            Source docBookSource = null;
            InputStream is = null;
            try
            {
                docBookSource = _sourceResolver.resolveURI(_FILLING_DATA_FOLDER_URI + "richtextWithForm.xml");
                is = docBookSource.getInputStream();
                text.setInputStream(is);
            }
            catch (IOException e)
            {
                getLogger().warn("Can't access richtext demonstrator docbook", e);
            }
            finally
            {
                _sourceResolver.release(docBookSource);
                IOUtils.closeQuietly(is);
            }
            
            ContentType articleContentType = _cTypeEP.getExtension("org.ametys.web.default.Content.article");
            content = _contentCreationHelper.initializeContent(articleContentType, sitemap, contentName);
            content = _contentCreationHelper.editContent(content, articleContentType, Map.of("content", text, "title", title));
        }
        return content;
    }

    private Content _getOrCreateDocbookDemonstratorContent(Sitemap sitemap) throws WorkflowException
    {
        String title = "Démonstrateur Docbook";
        String contentName = "demonstrateur-docbook";
        
        WorkflowAwareContent content = _contentCreationHelper.getContent(contentName, sitemap);
        
        if (content == null)
        {
            RichText text = new RichText();
            Source docBookSource = null;
            InputStream is = null;
            try
            {
                docBookSource = _sourceResolver.resolveURI(_FILLING_DATA_FOLDER_URI + "richtextDemonstrator.xml");
                is = docBookSource.getInputStream();
                text.setInputStream(is);
            }
            catch (IOException e)
            {
                getLogger().warn("Can't access richtext demonstrator docbook", e);
            }
            finally
            {
                IOUtils.closeQuietly(is);
                _sourceResolver.release(docBookSource);
            }
            text.setEncoding("UTF-8");
            text.setLastModificationDate(ZonedDateTime.now());
            text.setMimeType("text/xml");
            
            ContentType articleContentType = _cTypeEP.getExtension("org.ametys.web.default.Content.article");
            content = _contentCreationHelper.initializeContent(articleContentType, sitemap, contentName);
            content = _contentCreationHelper.editContent(content, articleContentType, Map.of("content", text, "title", title));
        }
        return content;
    }

    /**
     * Create Contents based on definition provided by the user
     * @param sitemap the sitemap to use for the contents
     * @return the Set of {@link ContentType} id that were created by the method
     */
    private Pair<Set<String>, Collection<Content>> _getOrCreateUserDefinedContents(Sitemap sitemap)
    {
        Set<String> overideContentTypeIds = new HashSet<>();
        Collection<Content> userContents = new HashSet<>();
        TraversableSource contentDir = null;
        try
        {
            contentDir = (TraversableSource) _sourceResolver.resolveURI("context:/" + _FILE_DIRECTORY_CONTENTS_URI);
            if (!contentDir.exists())
            {
                return Pair.of(overideContentTypeIds, Set.of());
            }
            if (!contentDir.isCollection())
            {
                throw new IllegalStateException("Root folder for the user defined content must be a folder");
            }
            for (TraversableSource contentSource : (Collection<TraversableSource>) contentDir.getChildren())
            {
                if (contentSource.isCollection())
                {
                    continue;
                }
                try
                {
                    Content content = _getOrCreateContentFromFile(contentSource, sitemap);
                    userContents.add(content);
                    String contentTypeId = content.getTypes()[0];
                    overideContentTypeIds.add(contentTypeId);
                    getLogger().info("A user defined configuration for contentType {} was found.", contentTypeId);
                }
                catch (IllegalStateException | ParserConfigurationException | SAXException | WorkflowException | IOException e)
                {
                    getLogger().error("Failed create content from source {}", contentSource.getURI(), e);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().info("No folder of user defined content for datafiller in the instance. Continuing with automatically generated content only.", e);
        }
        finally
        {
            _sourceResolver.release(contentDir);
        }
        return Pair.of(overideContentTypeIds, userContents);
    }

    /**
     * Create a content from a configuration file
     * @param source formated as an XML export
     * @param sitemap the sitemap where the content will be inserted
     * @return the newly created content
     * @throws ParserConfigurationException sent when failed to get the parser
     * @throws IOException sent when failed to read the source
     * @throws SourceNotFoundException sent when the source can't be found
     * @throws SAXException sent when the file is mal formed
     * @throws WorkflowException if an error occured while initializing the content
     */
    private Content _getOrCreateContentFromFile(Source source, Sitemap sitemap) throws ParserConfigurationException, SourceNotFoundException, IOException, SAXException, WorkflowException
    {
        String fileName = StringUtils.substringAfterLast(source.getURI(), "/");
        String name = StringUtils.substringBefore(fileName, ".xml");
        String contentTypeId = _getContentTypeIdForSource(source);
        
        ContentType contentType = _cTypeEP.getExtension(contentTypeId);
        if (contentType == null)
        {
            throw new IllegalStateException("Can't find content type for id: " + contentTypeId);
        }
        else
        {
            DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docFactory.newDocumentBuilder();
    
            try (InputStream is = source.getInputStream())
            {
                Document doc = docBuilder.parse(is);
                Element root = doc.getDocumentElement();
    
                // append the sitemapName to the content name to avoid name conflict between sitemap
                String sitemapName = sitemap.getSitemapName();
                String contentName = String.join("-", StringUtils.substringAfterLast(name, "."), sitemapName);
    
                // FIXME DATAFILLER-14 Support multiple content defined in a single file
                Element contentElement = DOMUtils.getChildElementByTagName(root, "content");
                Path imagesDir = Path.of(_context.getRealPath(_FILE_DIRECTORY_CONTENTS_URI + "/images"));
                ResourcesAdditionalDataGetter additionalDataGetter = new ResourcesAdditionalDataGetter(imagesDir);
                Content content = _getOrCreateContentFromXML(contentElement, sitemap, sitemapName, contentType, contentName, additionalDataGetter);
                return content;
            }
        }
    }

    /**
     * Create content from an Element describing the content
     * @param root the element root
     * @param sitemap the sitemap of the element
     * @param language the language
     * @param contentType the content type
     * @param contentName the content name
     * @param additionalDataGetter the {@link XMLValuesExtractorAdditionalDataGetter} used to find images
     * @return the created content
     * @throws WorkflowException thrown when failed to initialize the content
     */
    private Content _getOrCreateContentFromXML(Element root, Sitemap sitemap, String language, ContentType contentType, String contentName, ResourcesAdditionalDataGetter additionalDataGetter) throws WorkflowException
    {
        WorkflowAwareContent content = _contentCreationHelper.getContent(contentName, sitemap);
    
        if (content == null)
        {
            content = _contentCreationHelper.initializeContent(contentType, sitemap, contentName);
            ModelAwareXMLValuesExtractor extractor = new ModelAwareXMLValuesExtractor(root, additionalDataGetter, content.getModel());
            Map<String, Object> values;
            try
            {
                values = extractor.extractValues();
                content = _contentCreationHelper.editContent(content, contentType, values);
            }
            catch (Exception e)
            {
                getLogger().warn("Failed to extract values from XML for {}", contentType.getId(), e);
            }
    
            List<String> tags = new ArrayList<>();
            Element tagsElement = DOMUtils.getChildElementByTagName(root, "tags");
            if (tagsElement != null)
            {
                List<Element> tagElements = DOMUtils.getChildElements(tagsElement);
                for (Element tagElement : tagElements)
                {
                    tags.add(tagElement.getTagName());
                }
    
                if (!tags.isEmpty())
                {
                    _contentDAO.tag(List.of(content.getId()), tags, TagMode.INSERT, new HashMap<>(), true);
                }
            }
        }
        return content;
    }

    /** Get the contentType for a XML content configuration file.
     * @param source the configuration file
     * @return the contentTypeId String
     */
    private String _getContentTypeIdForSource(Source source)
    {
        // FIXME DATAFILLER-17 Use contentType markup instead of filename for Source
        String fileName = StringUtils.substringAfterLast(source.getURI(), "/");
        String name = StringUtils.substringBefore(fileName, ".xml");
        String contentTypeId = StringUtils.contains(name, "-") ? StringUtils.substringBefore(name, "-") : name;
        return contentTypeId;
    }

    /**
     * Get a previously created generic content of the given contentType in the sitemap or else,
     * create it based on contentType filling it with dummy values
     * 
     * @param sitemap the sitemap where the content will be defined
     * @param contentType the type of content to create
     * @return the created content
     * @throws WorkflowException if an error occured while initializing the content
     */
    private Content _getOrCreateGenericContent(ContentType contentType, Sitemap sitemap) throws WorkflowException
    {
        String contentName = contentType.getId();
        
        WorkflowAwareContent content = _contentCreationHelper.getContent(contentName, sitemap);
        
        if (content == null)
        {
            content = _contentCreationHelper.initializeContent(contentType, sitemap, contentName);
            View view = org.ametys.runtime.model.ViewHelper.getTruncatedView(View.of(contentType));
            Map<String, Object> values = _contentCreationHelper.getGenericValuesForView(view);
            content = _contentCreationHelper.editContent(content, contentType, values);
        }

        return content;
    }

    private Content _getOrCreateGenericContentWithoutIllustration(ContentType contentType, Sitemap sitemap) throws AmetysRepositoryException, WorkflowException
    {
        String contentName = contentType.getId() + "-no-illustration";
        
        ModifiableWorkflowAwareContent content = (ModifiableWorkflowAwareContent) _contentCreationHelper.getContent(contentName, sitemap);
        
        if (content == null)
        {
            content = (ModifiableWorkflowAwareContent) _contentCreationHelper.initializeContent(contentType, sitemap, contentName);
            View view = org.ametys.runtime.model.ViewHelper.getTruncatedView(View.of(contentType));
            Map<String, Object> values = _contentCreationHelper.getGenericValuesForView(view);
            content = (ModifiableWorkflowAwareContent) _contentCreationHelper.editContent(content, contentType, values);
            content.setTitle(content.getTitle() + " (no illustration)");
            content.removeValue("illustration");
            content.saveChanges();
        }

        return content;
    }
    
    /**
     * Create multiple pages to display the content in all possible view and
     * zone of the skin possible
     * 
     * @param rootPage the root page
     * @param content the content
     * @throws Exception if an error occurred
     */
    private void _createPagesForContent(Page rootPage, Content content) throws Exception
    {
        boolean noIllustration = StringUtils.contains(content.getName(), "-no-illustration");
        
        String contentTypeId = content.getTypes()[0];
        ContentType contentType = _cTypeEP.getExtension(contentTypeId);

        Site site = rootPage.getSite();
        String language = rootPage.getSitemapName();
        
        Skin skin = _skinsManager.getSkin(site.getSkinId());
        Set<String> templates = skin.getTemplates();
        boolean useSearchTemplateForSection = templates.contains("search");
        for (String templateId : templates)
        {
            if (_canCreateTemplate(rootPage.getId(), templateId, templateId))
            {
                // FIXME DATAFILLER-15 Test pages are not created for template with restriction
                String templateLabel = _i18nUtils.translate(skin.getTemplate(templateId).getLabel(), language);
                ModifiablePage templateSection = _sitemapPopulator.getOrCreatePage(rootPage, StringUtils.defaultIfBlank(templateLabel, templateId));
                // We set the template on the page to be able to determine the allowed zone.
                // We will remove the template later
                _pageDAO.setTemplate(List.of(templateSection.getId()), templateId, false);
                
                SkinTemplate template = skin.getTemplate(templateId);
                String defaultZoneId = template.getDefaultZoneId();
                for (SkinTemplateZone skinZone : template.getZones().values())
                {
                    if (_cTypeHandler.getAvailableContentTypes(templateSection, skinZone.getId()).contains(contentType.getId()))
                    {
                        // Create contents section
                        I18nizableText contentsTitle = new I18nizableText("plugin.data-filler", "PLUGINS_DATA_FILLER_CREATE_TEST_PAGE_CONTENTS_PAGE_TITLE");
                        ModifiablePage contentsSection = _sitemapPopulator.getOrCreatePage(templateSection, _i18nUtils.translate(contentsTitle, language));
        
                        // Create content section
                        String contentTitle = content.getTitle();
                        contentTitle = StringUtils.substringBefore(contentTitle, " (no illustration)");
                        ModifiablePage contentSection = _sitemapPopulator.getOrCreatePage(contentsSection, contentTitle);
                        if (contentSection.getTemplate() == null)
                        {
                            _pageDAO.setTemplate(List.of(contentSection.getId()), useSearchTemplateForSection ? "search" : templateId, false);
                        }

                        // If there is nothing in the zone already, we add a search service
                        if (contentSection.hasZone(defaultZoneId))
                        {
                            contentSection.getZone(defaultZoneId).remove();
                        }
                        Map<String, Object> serviceParam = new HashMap<>();
                        serviceParam.put("returnables", ContentReturnable.class.getName());
                        serviceParam.put("contentTypes", contentType.getId());
                        serviceParam.put("contexts", "{\"sites\":\"{\\\"context\\\":\\\"CURRENT_SITE\\\",\\\"sites\\\":[]}\",\"search-sitemap-context\":\"{\\\"context\\\":\\\"CHILD_PAGES\\\",\\\"page\\\":null}\",\"context-lang\":\"CURRENT\",\"tags\":[]}");
                        serviceParam.put("criteria", "{}");
                        serviceParam.put("launchSearchAtStartup", false);
                        serviceParam.put("initialSorts", "{\"name\":\"common$pertinence\",\"sort\":\"DESC\"}");
                        serviceParam.put("rightCheckingMode", "fast");
                        serviceParam.put("contentView", contentType.getView("abstract") != null ? "abstract" : "main");
                        serviceParam.put("resultPlace", "ABOVE_CRITERIA");
                        serviceParam.put("rss", false);
                        serviceParam.put("xslt", "pages/services/search/search-default.xsl");
                        _zoneItemManager.addService(contentSection.getId(), defaultZoneId, "org.ametys.web.service.SearchService", serviceParam);
                        
                        String viewName = "main";
                        // Create the zone page
                        String zoneLabel = _i18nUtils.translate(skinZone.getLabel(), language);
                        String pageName = StringUtils.defaultIfBlank(zoneLabel, skinZone.getId()) + (noIllustration ? " (no illustration)" : "");
                        ModifiablePage zonePage = _sitemapPopulator.getOrCreatePage(contentSection, pageName);
                        
                        List<String> pageIds = new ArrayList<>();
                        pageIds.add(zonePage.getId());
                        _pageDAO.setTemplate(pageIds, templateId, false);
                        
                        // Add the content in the zone if there is none already
                        if (zonePage.hasZone(skinZone.getId()))
                        {
                            zonePage.getZone(skinZone.getId()).remove();
                        }
                        _zoneDAO.addSharedContent(zonePage.getId(), skinZone.getId(), content.getId(), viewName);
                        
                        // If the zone is not the default zone, we also add the content in the default zone to improve the rendering
                        if (!skinZone.getId().equals(defaultZoneId) && _cTypeHandler.getAvailableContentTypes(zonePage, defaultZoneId).contains(contentType.getId()))
                        {
                            if (zonePage.hasZone(defaultZoneId))
                            {
                                zonePage.getZone(defaultZoneId).remove();
                            }
                            _zoneDAO.addSharedContent(zonePage.getId(), defaultZoneId, content.getId(), viewName);
                        }
                    }
                }
                // remove the template from the template page to validate the pages
                _pageDAO.setBlank(List.of(templateSection.getId()));
            }
        }
    }

    /**
     * True if we can create a page with template id
     * @param parentId the parent page id
     * @param pageTitle the page title
     * @param templateId the template id
     * @return True if we can create a page with template id
     */
    private boolean _canCreateTemplate(String parentId, String pageTitle, String templateId)
    {
        List<Map<String, Object>> availableTemplatesForCreation = _pageDAO.getAvailableTemplatesForCreation(null, parentId, pageTitle);
        return availableTemplatesForCreation.parallelStream().filter(map -> templateId.equals(map.get("id"))).findAny().isPresent();
    }

    class ResourcesAdditionalDataGetter implements XMLValuesExtractorAdditionalDataGetter
    {
        private Path _path;

        ResourcesAdditionalDataGetter(Path path)
        {
            _path = path;
        }

        @Override
        public Optional<Object> getAdditionalData(String dataPath, ElementType type)
        {
            Map<String, InputStream> fileData = new HashMap<>();
            if (Files.isDirectory(_path))
            {
                try (Stream<Path> files = Files.list(_path))
                {
                    files.forEach(LambdaUtils.wrapConsumer(file -> _initFile(file, fileData)));
                }
                catch (IOException e)
                {
                    throw new UncheckedIOException(e);
                }
            }
            return Optional.of(fileData);
        }

        private void _initFile(Path file, Map<String, InputStream> fileData) throws IOException
        {
            String fileName = file.getFileName().toString();
            InputStream is = AutoCloseInputStream.builder().setPath(file).get();
            fileData.put(fileName, is);
        }
    }
}
