/**
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormDirectoryDAO;
import org.ametys.plugins.forms.dao.FormPageDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.question.autofill.CurrentUserAutoFillSource;
import org.ametys.plugins.forms.question.computing.CostComputingType;
import org.ametys.plugins.forms.question.sources.ManualSourceType;
import org.ametys.plugins.forms.question.sources.ManualWithCostsSourceType;
import org.ametys.plugins.forms.question.types.AutocompleteAwareQuestionType;
import org.ametys.plugins.forms.question.types.ConfidentialAwareQuestionType;
import org.ametys.plugins.forms.question.types.DescriptibleAwareQuestionType;
import org.ametys.plugins.forms.question.types.MandatoryAwareQuestionType;
import org.ametys.plugins.forms.question.types.MultipleAwareQuestionType;
import org.ametys.plugins.forms.question.types.impl.CheckBoxQuestionType;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.impl.ComputedQuestionType;
import org.ametys.plugins.forms.question.types.impl.DateTimeQuestionType;
import org.ametys.plugins.forms.question.types.impl.FileQuestionType;
import org.ametys.plugins.forms.question.types.impl.MatrixQuestionType;
import org.ametys.plugins.forms.question.types.impl.NumberQuestionType;
import org.ametys.plugins.forms.question.types.impl.RichTextQuestionType;
import org.ametys.plugins.forms.question.types.impl.SimpleTextQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.ModifiableZoneItem;

/**
 * Manage the creation of generic forms for test purposes
 * Those forms will be integrated inside existing services.
 */
public class GenericFormCreationManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = GenericFormCreationManager.class.getName();
    private FormDAO _formDAO;
    private FormDirectoryDAO _formDirectoryDAO;
    private FormPageDAO _formPageDAO;
    private FormQuestionDAO _formQuestionDAO;
    private AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _formQuestionDAO = (FormQuestionDAO) manager.lookup(FormQuestionDAO.ROLE);
        _formPageDAO = (FormPageDAO) manager.lookup(FormPageDAO.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _formDirectoryDAO = (FormDirectoryDAO) manager.lookup(FormDirectoryDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Create a demo form and add it to services created for test purpuses, then create a form to test mini-survey
     * @param sitename the current site's name
     * @throws Exception exception thrown while creating form
     */
    public void createGenericForms(String sitename) throws Exception
    {
        Form createdForm = _createDemoForm(sitename);
        Form surveyForm = _createSurveyForm(sitename);
        
        _fillFormsInService(createdForm, surveyForm);
    }

    private void _fillFormsInService(Form createdForm, Form surveyForm)
    {
        String query = "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.forms.service.Display']";
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = _resolver.query(query);
        for (ModifiableZoneItem zoneItem : zoneItems)
        {
            ModifiableModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
            if (!serviceParameters.hasValue("formId"))
            {
                if (zoneItem.getZone().getName().equals("right"))
                {
                    serviceParameters.setValue("xslt", "pages/services/display/mini-survey.xsl");
                    serviceParameters.setValue("formId", surveyForm.getId());
                }
                else
                {
                    serviceParameters.setValue("formId", createdForm.getId());
                }
                zoneItem.saveChanges();
            }
        }
    }
    
    private Form _createDemoForm(String sitename) throws Exception
    {
        Map<String, Object> rootProperties = _formDirectoryDAO.getFormDirectoryProperties(_formDirectoryDAO.getFormDirectoriesRootNode(sitename), false);
        Map<String, String> createFormResult = _formDAO.createForm(sitename, (String) rootProperties.get("id"), "Demande travaux reprographie");
        String formId = createFormResult.get("id");
        Form form = _resolver.resolveById(formId);
        List<FormPage> pages = form.getPages();
        String page1Id = pages.get(0).getId();
        
        _createFullNameAutoFillSimpleText(page1Id);
        _createEmailRegexSimpleText(page1Id);
        _createBasicSimpleText(page1Id);
        
        // 2nd page
        Map<String, String> createPage2Result = _formPageDAO.createPage(formId, "Page");
        String page2Id = createPage2Result.get("id");
        
        _createFile(page2Id);
        _createNumber(page2Id);
        _createSimpleCombobox(page2Id);
        _createMultipleCombobox(page2Id);
        _createRadioGroup(page2Id);
        _createCheckBoxGroup(page2Id);
        _createUniqueCheckBox(page2Id);
        _createComputedCostField(page2Id);
        _createDateField(page2Id);
        _createBasicTextArea(page2Id);
        
        // 3rd page
        Map<String, String> createPageResult = _formPageDAO.createPage(formId, "Page");
        String page3Id = createPageResult.get("id");
        
        _createRichText(page3Id);
        _createSimpleMatrix(page3Id);
        _createMultipleMatrix(page3Id);
        _createDateTime(page3Id);
        
        form.saveChanges();
        
        _formDAO.setWorkflow(formId, "entry-form-default");
        return form;
    }
    
    private Form _createSurveyForm(String sitename) throws Exception
    {
        Map<String, Object> rootProperties = _formDirectoryDAO.getFormDirectoryProperties(_formDirectoryDAO.getFormDirectoriesRootNode(sitename), false);
        Map<String, String> createFormResult = _formDAO.createForm(sitename, (String) rootProperties.get("id"), "Mini-sondage");
        String formId = createFormResult.get("id");
        Form form = _resolver.resolveById(formId);
        List<FormPage> pages = form.getPages();
        String firstFormPageId = pages.get(0).getId();
        
        FormQuestion radioGroup = _createBasicField(firstFormPageId, "form.ChoicesList", "Que pensez-vous du désherbage à l'eau chaude comme alternative à l'utilisateion du glyphosate?", "Sélectionner une réponse en cochant la case correspondante", true);
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_FORMAT, ChoicesListQuestionType.CHECKBOX_FORMAT_VALUE);
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE, "org.ametys.plugins.forms.question.sources.Manual");
        radioGroup.setValue(ManualSourceType.ATTRIBUTE_GRID, "{"
                + "\"Je ne connais pas cette alternative\": {\"value\": \"a\"},"
                + "\"Je ne suis pas convaincu\": {\"value\": \"b\"},"
                + "\"Je suis convaincu\": {\"value\": \"c\"}"
                + "}");
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_OTHER, false);
        
        form.limitToOneEntryByUser(true);
        form.saveChanges();
        return form;
    }

    private void _createDateTime(String pageId)
    {
        FormQuestion dateTime = _createBasicField(pageId, "form.DateTime", "Date de la demande", "", false);
        dateTime.setValue(DateTimeQuestionType.ATTRIBUTE_DATE_FORMAT, DateTimeQuestionType.DATE_TIME_FORMAT_VALUE);
        dateTime.setValue(DateTimeQuestionType.ATTRIBUTE_MIN_DATETIME, ZonedDateTime.parse("2024-06-19T22:30:00.000Z"));
        dateTime.setValue(DateTimeQuestionType.ATTRIBUTE_MAX_DATETIME, ZonedDateTime.parse("2024-06-30T22:30:00.000Z"));
    }

    private void _createMultipleMatrix(String pageId)
    {
        FormQuestion matrix = _createBasicField(pageId, "form.Matrix", "Mode de livraison", "La description du champ matrice", false);
        matrix.setValue(MatrixQuestionType.ATTRIBUTE_GRID, "{"
            + "\"options\":"
            + "{\"opt_value\":\"Lundi\","
            + "\"opt_value_1\":\"Mardi\","
            + "\"opt_value_2\":\"Mercredi\","
            + "\"opt_value_3\":\"Jeudi\","
            + "\"opt_value_4\":\"Vendredi\"},"
            + "\"columns\":{"
            + "\"col_value\":\"Matin\","
            + "\"col_value_1\":\"Après-midi\"}}");
        matrix.setValue(MultipleAwareQuestionType.ATTRIBUTE_MULTIPLE, true);
    }

    private void _createSimpleMatrix(String pageId)
    {
        FormQuestion matrix = _createBasicField(pageId, "form.Matrix", "Mode de livraison", "La description du champ matrice", false);
        matrix.setValue(MatrixQuestionType.ATTRIBUTE_GRID, "{"
                + "\"options\":"
                + "{\"opt_value\":\"À domicile\","
                + "\"opt_value_1\":\"En point relai\"},"
                + "\"columns\":{"
                + "\"col_value\":\"Rapide (2 jours ouvrés)\","
                + "\"col_value_1\":\"Normale (3-5 jours)\"}}");
        matrix.setValue(MultipleAwareQuestionType.ATTRIBUTE_MULTIPLE, false);
    }

    private void _createComputedCostField(String pageId)
    {
        Map<String, Object> computingResult = _formQuestionDAO.createQuestion(pageId, "form.Computed");
        FormQuestion computing = _resolver.resolveById((String) computingResult.get("id"));
        computing.setValue(ComputedQuestionType.ATTRIBUTE_COMPUTING_TYPE, "org.ametys.plugins.forms.question.computing.Cost");
        computing.setValue(FormQuestion.ATTRIBUTE_TITLE, "Coût de la reprographie");
        computing.setValue(CostComputingType.ATTRIBUTE_CURRENCY, CostComputingType.EURO_CURRENCY_VALUE);
        computing.setValue(ConfidentialAwareQuestionType.ATTRIBUTE_CONFIDENTIALITY, false);
    }

    private void _createRichText(String pageId)
    {
        Map<String, Object> richTextResult = _formQuestionDAO.createQuestion(pageId, "form.RichText");
        FormQuestion richText = _resolver.resolveById((String) richTextResult.get("id"));
        String text = "<p><em>What is Lorem Ipsum?</em></p>"
                + "<p><strong>Lorem Ipsum</strong> is simply dummy text of the printing and typesetting industry.</p>"
                + "<ol class=\"arabic\">"
                + "<li>Lorem Ipsum has been the industry's standard <strong>dummy text</strong> ever since the 1500s, when an unknown printer took a galley of type and scrambled it to make a type specimen book.</li>"
                + "<li>It has survived not only five centuries, but also the leap into electronic typesetting, remaining essentially unchanged.</li>"
                + "<li>It was popularised in the 1960s with the release of Letraset sheets containing Lorem Ipsum passages, and more recently with desktop publishing software like Aldus PageMaker including versions of Lorem Ipsum.</li>"
                + "</ol>)";
        ModelItem modelItem = richText.getType().getModel().getModelItem(RichTextQuestionType.ATTRIBUTE_RICH_TEXT);
        if (modelItem instanceof ElementDefinition definition)
        {
            ElementType type = definition.getType();
            Object typedValue = type.fromJSONForClient(text, DataContext.newInstance().withDataPath(RichTextQuestionType.ATTRIBUTE_RICH_TEXT));
            richText.setValue(RichTextQuestionType.ATTRIBUTE_RICH_TEXT, typedValue);
        }
        richText.setValue(FormQuestion.ATTRIBUTE_TITLE, "Champs de test complémentaires");
        richText.setValue(DescriptibleAwareQuestionType.ATTRIBUTE_DESCRIPTION, "la description du champ texte riche");
    }

    private void _createDateField(String pageId)
    {
        FormQuestion date = _createBasicField(pageId, "form.DateTime", "Date de retour souhaitée", "", false);
        date.setValue(DateTimeQuestionType.ATTRIBUTE_DATE_FORMAT, DateTimeQuestionType.DATE_FORMAT_VALUE);
        date.setValue(DateTimeQuestionType.ATTRIBUTE_MIN_DATE, LocalDate.parse("2024-01-01"));
        date.setValue(DateTimeQuestionType.ATTRIBUTE_MAX_DATE, LocalDate.parse("2024-12-12"));
    }

    private void _createUniqueCheckBox(String pageId)
    {
        Map<String, Object> checkBoxResult = _formQuestionDAO.createQuestion(pageId, "form.Checkbox");
        FormQuestion checkbox = _resolver.resolveById((String) checkBoxResult.get("id"));
        checkbox.setValue(FormQuestion.ATTRIBUTE_TITLE, "Impression recto/verso");
        checkbox.setValue(CheckBoxQuestionType.ATTRIBUTE_IS_CHECKED, true);
        checkbox.setValue(DescriptibleAwareQuestionType.ATTRIBUTE_DESCRIPTION, "la description de la case à cocher");
    }

    private void _createCheckBoxGroup(String pageId)
    {
        FormQuestion checkboxGroup = _createBasicField(pageId, "form.ChoicesList", "Type de papier", "Description du groupe de cases à cocher", false);
        checkboxGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_FORMAT, ChoicesListQuestionType.CHECKBOX_FORMAT_VALUE);
        checkboxGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE, "org.ametys.plugins.forms.question.sources.Manual");
        checkboxGroup.setValue(ManualSourceType.ATTRIBUTE_GRID, "{"
                + "\"Photo\": {\"value\": \"photo\"},"
                + "\"Adhésif\": {\"value\": \"adh_sif\"},"
                + "\"Simple\": {\"value\": \"simple\"},"
                + "\"Cartonné\": {\"value\": \"carton\"}"
                + "}");
        checkboxGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_OTHER, true);
        checkboxGroup.setValue(MultipleAwareQuestionType.ATTRIBUTE_MULTIPLE, true);
    }

    private void _createRadioGroup(String pageId)
    {
        FormQuestion radioGroup = _createBasicField(pageId, "form.ChoicesList", "Couleur", "Description du groupe radio", false);
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_FORMAT, ChoicesListQuestionType.CHECKBOX_FORMAT_VALUE);
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE, "org.ametys.plugins.forms.question.sources.Manual");
        radioGroup.setValue(ManualSourceType.ATTRIBUTE_GRID, "{"
                + "\"Noir et blanc\": {\"value\": \"noir_et_blanc\"},"
                + "\"Couleur\": {\"value\": \"couleur\"}"
                + "}");
        radioGroup.setValue(ChoicesListQuestionType.ATTRIBUTE_OTHER, false);
    }

    private void _createMultipleCombobox(String pageId)
    {
        FormQuestion multiCombobox = _createBasicField(pageId, "form.ChoicesList", "Format papier de la copie", "Si autre format, précisez les dimensions en cm", false);
        multiCombobox.setValue(ChoicesListQuestionType.ATTRIBUTE_FORMAT, ChoicesListQuestionType.COMBOBOXBOX_FORMAT_VALUE);
        multiCombobox.setValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE, "org.ametys.plugins.forms.question.sources.ManualWithCosts");
        multiCombobox.setValue(ManualWithCostsSourceType.ATTRIBUTE_GRID_COSTS, "{"
                + "\"A1 (59,4 x 84,1 cm)\": {\"value\": \"a1__59_4_x_84_1_cm_\", \"cost\":0.25},"
                + "\"A2 (42 x 59,4 cm)\": {\"value\": \"a2__42_59_4_cm_\", \"cost\":0.30},"
                + "\"A3 (29,7 x 42 cm)\": {\"value\": \"a3__29_7_x_42_cm_\", \"cost\":0.40},"
                + "\"A4 (21 x 29,7 cm)\": {\"value\": \"a4__21_x_29_7_cm_\", \"cost\":0.50},"
                + "\"A5 (14,8 x 21 cm)\": {\"value\": \"a5__14_8_x_21_cm_\", \"cost\":1.50}"
                + "}");
        multiCombobox.setValue(ManualSourceType.ATTRIBUTE_GRID, _getChoiceGrid());
        multiCombobox.setValue(MultipleAwareQuestionType.ATTRIBUTE_MULTIPLE, true);
    }

    private void _createSimpleCombobox(String pageId)
    {
        FormQuestion combobox = _createBasicField(pageId, "form.ChoicesList", "Format papier original", "Description de ma liste à choix", false);
        combobox.setValue(ChoicesListQuestionType.ATTRIBUTE_FORMAT, ChoicesListQuestionType.COMBOBOXBOX_FORMAT_VALUE);
        combobox.setValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE, "org.ametys.plugins.forms.question.sources.Manual");
        combobox.setValue(ManualWithCostsSourceType.ATTRIBUTE_GRID_COSTS, "{"
                + "\"A1 (59,4 x 84,1 cm)\": {\"value\": \"a1__59_4_x_84_1_cm_\", \"cost\":0},"
                + "\"A2 (42 x 59,4 cm)\": {\"value\": \"a2__42_59_4_cm_\", \"cost\":0},"
                + "\"A3 (29,7 x 42 cm)\": {\"value\": \"a3__29_7_x_42_cm_\", \"cost\":0},"
                + "\"A4 (21 x 29,7 cm)\": {\"value\": \"a4__21_x_29_7_cm_\", \"cost\":0},"
                + "\"A5 (14,8 x 21 cm)\": {\"value\": \"a5__14_8_x_21_cm_\", \"cost\":0}"
                + "}");
        combobox.setValue(ManualSourceType.ATTRIBUTE_GRID, _getChoiceGrid());
        combobox.setValue(ChoicesListQuestionType.ATTRIBUTE_OTHER, true);
        combobox.setValue(MultipleAwareQuestionType.ATTRIBUTE_MULTIPLE, false);
    }

    private String _getChoiceGrid()
    {
        String choicesOptions = "{"
                + "\"A1 (59,4 x 84,1 cm)\": {\"value\": \"a1__59_4_x_84_1_cm_\"},"
                + "\"A2 (42 x 59,4 cm)\": {\"value\": \"a2__42_59_4_cm_\"},"
                + "\"A3 (29,7 x 42 cm)\": {\"value\": \"a3__29_7_x_42_cm_\"},"
                + "\"A4 (21 x 29,7 cm)\": {\"value\": \"a4__21_x_29_7_cm_\"},"
                + "\"A5 (14,8 x 21 cm)\": {\"value\": \"a5__14_8_x_21_cm_\"}"
                + "}";
        return choicesOptions;
    }

    private void _createNumber(String pageId)
    {
        FormQuestion number = _createBasicField(pageId, "form.Number", "Quantité totale", "Description de mon champ nombre", false);
        number.setValue(NumberQuestionType.ATTRIBUTE_NUMBER_TYPE, NumberQuestionType.DOUBLE_NUMBER_VALUE);
        number.setValue(NumberQuestionType.ATTRIBUTE_MIN_DOUBLE, 5.1);
        number.setValue(NumberQuestionType.ATTRIBUTE_MAX_DOUBLE, 49.3);
    }

    private void _createFile(String pageId)
    {
        FormQuestion file = _createBasicField(pageId, "form.File", "Maquette à joindre", "Description de mon champ fichier", false);
        file.setValue(FileQuestionType.ATTRIBUTE_EXTENSIONS, FileQuestionType.IMAGE_EXTENSIONS_VALUE);
        file.setValue(FileQuestionType.ATTRIBUTE_MAX_SIZE, 5);
    }

    private void _createEmailRegexSimpleText(String pageId)
    {
        FormQuestion textEmail = _createBasicField(pageId, "form.SimpleText", "Email demandeur", "", false);
        textEmail.setValue(SimpleTextQuestionType.ATTRIBUTE_REGEXP, "email");
        textEmail.setValue(SimpleTextQuestionType.ATTRIBUTE_PLACEHOLDER, "toto@exemple.fr");
    }

    private void _createFullNameAutoFillSimpleText(String pageId)
    {
        FormQuestion textUser = _createBasicField(pageId, "form.SimpleText", "Nom demandeur", "la description du champ texte", true);
        textUser.setValue(SimpleTextQuestionType.ATTRIBUTE_AUTOFILL_SOURCE, "org.ametys.plugins.forms.question.autofill.Default");
        textUser.setValue(CurrentUserAutoFillSource.ATTRIBUTE_AUTOFILL, "fullName");
        textUser.setValue(AutocompleteAwareQuestionType.ATTRIBUTE_AUTOCOMPLETE, "name");
    }
    
    private FormQuestion _createBasicSimpleText(String pageId)
    {
        return _createBasicField(pageId, "form.SimpleText", "Service demandeur", "", false);
    }
    
    private FormQuestion _createBasicTextArea(String pageId)
    {
        return _createBasicField(pageId, "form.TextArea", "Précisions complémentaires", "La description du champ zone de texte", false);
    }
    
    private FormQuestion _createBasicField(String pageId, String type, String title, String desc, boolean isMandatory)
    {
        Map<String, Object> fieldResult = _formQuestionDAO.createQuestion(pageId, type);
        FormQuestion field = _resolver.resolveById((String) fieldResult.get("id"));
        field.setValue(MandatoryAwareQuestionType.ATTRIBUTE_MANDATORY, isMandatory);
        field.setValue(FormQuestion.ATTRIBUTE_TITLE, title);
        field.setValue(DescriptibleAwareQuestionType.ATTRIBUTE_DESCRIPTION, desc);
        return field;
    }
    
}
