/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.ServicesAssignmentHandler;
import org.ametys.web.repository.page.ZoneItemManager;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.service.Service;
import org.ametys.web.service.ServiceExtensionPoint;
import org.ametys.web.service.ServiceParameter;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinTemplateZone;
import org.ametys.web.skin.SkinsManager;

/**
 * Manage the creation of generic services for test purposes
 * Those service will be integrated inside new pages.
 */
public class GenericServiceCreationManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = GenericServiceCreationManager.class.getName();
    
    private static final String _SERVICES_PARAMETER_PLUGIN_DIRECTORY = "plugin:data-filler://filling-data/services";
    private static final String _FILE_DIRECTORY_SERVICES_URI = "/WEB-INF/param/datafill/services";

    /** The i18nUtils */
    private I18nUtils _i18nUtils;
    /** The page DAO */
    private PageDAO _pageDAO;
    /** The service extension point */
    private ServiceExtensionPoint _serviceEP;
    /** The service assigment handler */
    private ServicesAssignmentHandler _serviceHandler;
    /** The Sitemap creation Manager */
    private SitemapPopulator _sitemapPopulator;
    /** The skin manager */
    private SkinsManager _skinsManager;
    /** The source resolver */
    private SourceResolver _sourceResolver;
    /** The zone item manager */
    private ZoneItemManager _zoneItemManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _serviceEP = (ServiceExtensionPoint) manager.lookup(ServiceExtensionPoint.ROLE);
        _serviceHandler = (ServicesAssignmentHandler) manager.lookup(ServicesAssignmentHandler.ROLE);
        _sitemapPopulator = (SitemapPopulator) manager.lookup(SitemapPopulator.ROLE);
        _skinsManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _zoneItemManager = (ZoneItemManager) manager.lookup(ZoneItemManager.ROLE);
    }

    /**
     * Create services and page to test display of such services
     * @param sitemap new pages will be added to this sitemap
     * @return a set of all the created page that need attachements
     * @throws IOException if any IO error occurs
     */
    public Set<String> createGenericServicePages(Sitemap sitemap) throws IOException
    {
        Set<ServiceObject> createdServices = _createAllServices();
        
        ModifiablePage autogeneratedSection = _sitemapPopulator.getOrCreatePage(sitemap, "Test des gabarits");
        Set<String> pageWithAttachementId = new HashSet<>();
        for (ServiceObject service : createdServices)
        {
            pageWithAttachementId.addAll(_createPagesForService(autogeneratedSection, service));
        }
        return pageWithAttachementId;
    }
    
    /**
     * Create multiples services for test purposes based on configuration files
     * @return a Set of {@link ServiceObject} created by the method
     */
    private Set<ServiceObject> _createAllServices()
    {
        Set<ServiceObject> createdServices = new HashSet<>();
        // We have to hard code the list of file as we can't traverse a JAR source and list the content of a directory
        final Set<String> predefinedServices = Set.of(
                "org.ametys.plugins.linkdirectory.DirectoryService.xml",
                "org.ametys.web.service.FilteredContentsService.xml",
                "org.ametys.web.service.SitemapService.xml",
                "org.ametys.plugins.syndication.service.RSS.xml",
                "org.ametys.web.service.FrontSearchService.xml",
                "org.ametys.plugins.userdirectory.service.OrganizationChart.xml",
                "org.ametys.web.service.AttachmentsService.xml",
                "org.ametys.web.service.SearchService.xml",
                "org.ametys.web.service.SearchService-user.xml",
                "org.ametys.forms.service.Display.xml",
                "org.ametys.plugins.forms.workflow.service.dashboard.xml",
                "org.ametys.plugins.forms.workflow.service.admin.dashboard.xml");

        // Create user defined service
        TraversableSource servicesDir = null;
        try
        {
            servicesDir = (TraversableSource) _sourceResolver.resolveURI("context:/" + _FILE_DIRECTORY_SERVICES_URI);
            if (servicesDir.exists())
            {
                if (!servicesDir.isCollection())
                {
                    throw new IllegalStateException("Root folder for the user defined content must be a folder");
                }
                for (TraversableSource serviceSource : (Collection<TraversableSource>) servicesDir.getChildren())
                {
                    if (serviceSource.isCollection())
                    {
                        continue;
                    }
                    try
                    {
                        createdServices.add(_createServiceFromFile(serviceSource));
                    }
                    catch (IOException e)
                    {
                        getLogger().error("An error occurred: can't create service from file {}", serviceSource.getURI(), e);
                    }
                    catch (ParserConfigurationException | SAXException | IllegalStateException e)
                    {
                        getLogger().warn("An error occurred while parsing the file {}. Skipping...", serviceSource.getURI(), e);
                    }
                }
            }
        }
        catch (IOException e)
        {
            getLogger().info("No folder of user defined content for datafiller in the instance. Continuing with automatically generated content only.", e);
        }
        finally
        {
            _sourceResolver.release(servicesDir);
        }

        
        // Create services defined by the plugin.
        for (String service : predefinedServices)
        {
            Source serviceFile = null;
            try
            {
                serviceFile = _sourceResolver.resolveURI(_SERVICES_PARAMETER_PLUGIN_DIRECTORY + "/" + service);
                createdServices.add(_createServiceFromSource(serviceFile));
            }
            catch (IOException e)
            {
                getLogger().warn("An error occured while reading file {}. Skipping...", service, e);
            }
            catch (ParserConfigurationException | SAXException | IllegalStateException e)
            {
                getLogger().warn("An error occurred while parsing the file {}. Skipping...", service, e);
            }
            finally
            {
                _sourceResolver.release(serviceFile);
            }
        }
        return createdServices;
    }
    
    /**
     * Create service from file
     * @param serviceSource the file containing the description of the service and its parameters
     * @return the ServiceObject created
     * @throws ParserConfigurationException if no DocumentBuilder can be configured
     * @throws IOException if any IO error occurs
     * @throws SAXException if any parse error occurs
     * @throws IllegalStateException if no service is available for the id
     */
    private ServiceObject _createServiceFromFile(TraversableSource serviceSource) throws ParserConfigurationException, SAXException, IOException, IllegalStateException
    {
        String fileName = StringUtils.substringAfterLast(serviceSource.getURI(), "/");
        String name = StringUtils.substringBefore(fileName, ".xml");
        String serviceId = StringUtils.contains(name, "-") ? StringUtils.substringBefore(name, "-") : name;
        
        Service service = _serviceEP.getExtension(serviceId);
        if (service != null)
        {
            DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docFactory.newDocumentBuilder();
            
            try (InputStream is = serviceSource.getInputStream())
            {
                Document doc = docBuilder.parse(is);
                Element root = doc.getDocumentElement();

                Map<String, Object> serviceParameters = _getServiceParameters(root, service);
                return new ServiceObject(service, serviceParameters);
            }
        }
        else
        {
            throw new IllegalStateException("No service available for service Id : " + serviceId);
        }
        
    }
    
    /**
     * Create service from file
     * @param source the source containing the description of the service and its parameters
     * @return the created ServiceObject
     * @throws ParserConfigurationException if no DocumentBuilder can be configured
     * @throws IOException if any IO error occurs
     * @throws SAXException if any parse error occurs
     * @throws IllegalStateException if no service is available for the id
     */
    private ServiceObject _createServiceFromSource(Source source) throws ParserConfigurationException, SAXException, IOException, IllegalStateException
    {
        String fileName = StringUtils.substringAfterLast(source.getURI(), "/");
        String name = StringUtils.substringBefore(fileName, ".xml");
        String serviceId = StringUtils.contains(name, "-") ? StringUtils.substringBefore(name, "-") : name;
        
        Service service = _serviceEP.getExtension(serviceId);
        if (service != null)
        {
            DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docFactory.newDocumentBuilder();
            
            try (InputStream is = source.getInputStream())
            {
                Document doc = docBuilder.parse(is);
                Element root = doc.getDocumentElement();

                Map<String, Object> serviceParameters = _getServiceParameters(root, service);
                return new ServiceObject(service, serviceParameters);
            }
        }
        else
        {
            throw new IllegalStateException("No service available for service Id : " + serviceId);
        }
        
    }
    
    /**
     * Get the parameters from a service and set them with the setting in {@code root}
     * @param root root element containing all the service parameters
     * @param service the parameters Map will be retrieved from this service
     * @return the service parameters map
     */
    private Map<String, Object> _getServiceParameters(Element root, Service service)
    {
        Map<String, Object> serviceParameters = new HashMap<>();
        List<Element> serviceParameterElementList = DOMUtils.getChildElements(root);
        Map<String, ModelItem> parameters = service.getParameters();
        for (Element serviceParameterNode : serviceParameterElementList)
        {
            String parameterName = serviceParameterNode.getTagName();
            
            if (parameters.containsKey(parameterName))
            {
                ModelItem modelItem = parameters.get(parameterName);
                if (modelItem instanceof RepeaterDefinition)
                {
                    _handleRepeaterParameters(service, serviceParameters, serviceParameterNode, parameterName, modelItem);
                }
                else if (modelItem instanceof CompositeDefinition)
                {
                    _handleCompositeParameters(service, serviceParameters, serviceParameterNode, parameterName, modelItem);
                }
                else if (modelItem instanceof ServiceParameter)
                {
                    Object value = _getServiceParameter(serviceParameterNode, (ServiceParameter) modelItem);
                    if (value != null)
                    {
                        serviceParameters.put(parameterName, value);
                    }
                }
            }
            else
            {
                getLogger().warn("No parameter with name {} defined for service {}", parameterName, service.getId());
            }
        }
        
        return serviceParameters;
    }
    
    /**
     * Handle composite parameters
     * @param service the service
     * @param serviceParameters the service parameter
     * @param serviceParameterNode the service parameter node
     * @param parameterName the parameter name
     * @param modelItem the model item
     */
    private void _handleCompositeParameters(Service service, Map<String, Object> serviceParameters, Element serviceParameterNode, String parameterName, ModelItem modelItem)
    {
        CompositeDefinition compositeDef = (CompositeDefinition) modelItem;
        List<Element> childAttributeElements = DOMUtils.getChildElements(serviceParameterNode);
        for (Element childAttributeElement : childAttributeElements)
        {
            String childAttributeName = childAttributeElement.getTagName();
            ModelItem childModelItem = compositeDef.getChild(childAttributeName);
            if (childModelItem != null)
            {
                if (childModelItem instanceof ServiceParameter)
                {
                    Object value = _getServiceParameter(childAttributeElement, (ServiceParameter) childModelItem);
                    if (value != null)
                    {
                        String newParameterName = parameterName + "/" + childAttributeName;
                        serviceParameters.put(newParameterName, value);
                    }
                }
                else
                {
                    getLogger().error("Don't handle composite of composite...:(");
                }
            }
            else
            {
                getLogger().warn("No parameter with name {} defined for service {}", childAttributeName, service.getId());
            }
        }
    }

    /**
     * Handle repeater parameters
     * @param service the service
     * @param serviceParameters the service parameter
     * @param serviceParameterNode the service parameter node
     * @param parameterName the parameter name
     * @param modelItem the model item
     */
    private void _handleRepeaterParameters(Service service, Map<String, Object> serviceParameters, Element serviceParameterNode, String parameterName, ModelItem modelItem)
    {
        RepeaterDefinition repeaterDef = (RepeaterDefinition) modelItem;
        List<Element> entryElements = DOMUtils.getChildElementsByTagName(serviceParameterNode, "entry");
        serviceParameters.put("_" + parameterName + "/size", entryElements.size());
        for (int k = 0; k < entryElements.size(); k++)
        {
            Element repeaterElement = entryElements.get(k);
            List<Element> childAttributeElements = DOMUtils.getChildElements(repeaterElement);
            int entryPosition = k + 1;
            serviceParameters.put("_" + parameterName + "[" + entryPosition + "]/previous-position", -1);
            serviceParameters.put("_" + parameterName + "[" + entryPosition + "]/position", entryPosition);
            for (Element childAttributeElement : childAttributeElements)
            {
                ModelItem childModelItem = repeaterDef.getChild(childAttributeElement.getTagName());
                if (childModelItem != null)
                {
                    if (childModelItem instanceof ServiceParameter)
                    {
                        Object value = _getServiceParameter(childAttributeElement, (ServiceParameter) childModelItem);
                        if (value != null)
                        {
                            String newParameterName = parameterName + "[" + entryPosition + "]/" + childAttributeElement.getTagName();
                            serviceParameters.put(newParameterName, value);
                        }
                    }
                    else
                    {
                        getLogger().error("Don't handle repeater of repeater...:(");
                    }
                }
                else
                {
                    getLogger().warn("No parameter with name {} defined for service {}", childAttributeElement.getTagName(), service.getId());
                }
            }
        }
    }
    
    private Object _getServiceParameter(Element serviceParameterElement, ServiceParameter serviceParameter)
    {
        if (StringUtils.isNotBlank(serviceParameterElement.getTextContent()))
        {
            ElementType newType = serviceParameter.getType();
            if (serviceParameter.isMultiple())
            {
                return Arrays.asList(StringUtils.split(serviceParameterElement.getTextContent().trim(), "\n")).stream()
                                                                                                   .map(valueAsString -> newType.castValue(valueAsString.trim()))
                                                                                                   .collect(Collectors.toList());
            }
            else
            {
                return newType.castValue(serviceParameterElement.getTextContent());
            }
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Create multiple pages for a service to display in every zone of every template it is allowed to be inserted.
     * @param rootPage the root page
     * @param serviceObject the service object
     * @throws IOException if an error occurred while saving parameters
     */
    private Set<String> _createPagesForService(Page rootPage, ServiceObject serviceObject) throws IOException
    {
        Service service = serviceObject.getService();
        String serviceId = service.getId();
        
        Site site = rootPage.getSite();
        String language = rootPage.getSitemapName();

        Set<String> pageWithAttachementId = new HashSet<>();
        Skin skin = _skinsManager.getSkin(site.getSkinId());
        for (String templateId : skin.getTemplates())
        {
            if (_canCreateTemplate(rootPage.getId(), templateId, templateId))
            {
                SkinTemplate template = skin.getTemplate(templateId);
                // Create template page
                String templateLabel = _i18nUtils.translate(skin.getTemplate(templateId).getLabel(), language);
                ModifiablePage templatePage = _sitemapPopulator.getOrCreatePage(rootPage, StringUtils.defaultIfBlank(templateLabel, templateId));
                // We set the template on the page to be able to determine the allowed zone.
                // We will remove the template later
                _pageDAO.setTemplate(List.of(templatePage.getId()), templateId, false);
                
                for (SkinTemplateZone skinZone : template.getZones().values())
                {
                    if (_serviceHandler.getAvailableServices(templatePage, skinZone.getId()).contains(serviceId))
                    {
                        // Create services page
                        I18nizableText servicesTitle = new I18nizableText("plugin.data-filler", "PLUGINS_DATA_FILLER_CREATE_TEST_PAGE_SERVICES_PAGE_TITLE");
                        ModifiablePage servicesPage = _sitemapPopulator.getOrCreatePage(templatePage, _i18nUtils.translate(servicesTitle, rootPage.getSitemapName()));
                        
                        // Create service page
                        String servicePageTitle = (String) serviceObject.getParameters().get("header");
                        servicePageTitle = servicePageTitle == null ? _i18nUtils.translate(service.getLabel() , rootPage.getSitemapName()) : servicePageTitle;
                        ModifiablePage servicePage = _sitemapPopulator.getOrCreatePage(servicesPage, servicePageTitle);
                        
                        // Create the zone page
                        String zoneLabel = _i18nUtils.translate(skinZone.getLabel(), language);
                        ModifiablePage zonePage = _sitemapPopulator.getOrCreatePage(servicePage, StringUtils.defaultIfBlank(zoneLabel, skinZone.getId()));
                        
                        _pageDAO.setTemplate(List.of(zonePage.getId()), templateId, false);
                        _zoneItemManager.addService(zonePage.getId(), skinZone.getId(), serviceId, serviceObject.getParameters());
                        if (serviceId.equals("org.ametys.web.service.AttachmentsService"))
                        {
                            pageWithAttachementId.add(zonePage.getId());
                        }
                    }
                }
                // Remove the template
                _pageDAO.setBlank(List.of(templatePage.getId()));
            }
        }
        return pageWithAttachementId;
    }
    
    /**
     * True if we can create a page with template id
     * @param parentId the parent page id
     * @param pageTitle the page title
     * @param templateId the template id
     * @return True if we can create a page with template id
     */
    private boolean _canCreateTemplate(String parentId, String pageTitle, String templateId)
    {
        List<Map<String, Object>> availableTemplatesForCreation = _pageDAO.getAvailableTemplatesForCreation(null, parentId, pageTitle);
        return availableTemplatesForCreation.parallelStream()
                                            .filter(map -> templateId.equals(map.get("id")))
                                            .findAny()
                                            .isPresent();
    }
    
    /**
     * Simple Object to store service information
     */
    private class ServiceObject
    {
        private Service _service;
        private Map<String, Object> _parameters;
        
        public ServiceObject(Service service, Map<String, Object> parameters)
        {
            setService(service);
            setParameters(parameters);
        }

        public Service getService()
        {
            return _service;
        }

        public void setService(Service service)
        {
            this._service = service;
        }

        public Map<String, Object> getParameters()
        {
            return _parameters;
        }

        public void setParameters(Map<String, Object> parameters)
        {
            this._parameters = parameters;
        }
    }
}
