/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.exchange;

import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.EnumUtils;
import org.apache.commons.lang3.StringUtils;
import org.jsoup.Jsoup;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.SessionAttributeProvider;
import org.ametys.plugins.core.impl.authentication.FormCredentialProvider;
import org.ametys.plugins.messagingconnector.AbstractMessagingConnector;
import org.ametys.plugins.messagingconnector.CalendarEvent;
import org.ametys.plugins.messagingconnector.EmailMessage;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.messagingconnector.MessagingConnectorException;
import org.ametys.plugins.messagingconnector.MessagingConnectorException.ExceptionType;
import org.ametys.runtime.config.Config;
import org.ametys.web.WebSessionAttributeProvider;

import microsoft.exchange.webservices.data.core.ExchangeService;
import microsoft.exchange.webservices.data.core.enumeration.availability.AvailabilityData;
import microsoft.exchange.webservices.data.core.enumeration.misc.ConnectingIdType;
import microsoft.exchange.webservices.data.core.enumeration.misc.ExchangeVersion;
import microsoft.exchange.webservices.data.core.enumeration.property.LegacyFreeBusyStatus;
import microsoft.exchange.webservices.data.core.enumeration.property.MeetingResponseType;
import microsoft.exchange.webservices.data.core.enumeration.property.WellKnownFolderName;
import microsoft.exchange.webservices.data.core.enumeration.property.time.DayOfTheWeek;
import microsoft.exchange.webservices.data.core.enumeration.search.LogicalOperator;
import microsoft.exchange.webservices.data.core.enumeration.service.ConflictResolutionMode;
import microsoft.exchange.webservices.data.core.enumeration.service.DeleteMode;
import microsoft.exchange.webservices.data.core.enumeration.service.SendInvitationsMode;
import microsoft.exchange.webservices.data.core.enumeration.service.SendInvitationsOrCancellationsMode;
import microsoft.exchange.webservices.data.core.enumeration.service.ServiceResult;
import microsoft.exchange.webservices.data.core.exception.http.HttpErrorException;
import microsoft.exchange.webservices.data.core.exception.service.remote.ServiceRequestException;
import microsoft.exchange.webservices.data.core.exception.service.remote.ServiceResponseException;
import microsoft.exchange.webservices.data.core.response.AttendeeAvailability;
import microsoft.exchange.webservices.data.core.service.folder.CalendarFolder;
import microsoft.exchange.webservices.data.core.service.folder.Folder;
import microsoft.exchange.webservices.data.core.service.item.Appointment;
import microsoft.exchange.webservices.data.core.service.item.Item;
import microsoft.exchange.webservices.data.core.service.schema.EmailMessageSchema;
import microsoft.exchange.webservices.data.credential.ExchangeCredentials;
import microsoft.exchange.webservices.data.credential.WebCredentials;
import microsoft.exchange.webservices.data.misc.ImpersonatedUserId;
import microsoft.exchange.webservices.data.misc.availability.AttendeeInfo;
import microsoft.exchange.webservices.data.misc.availability.GetUserAvailabilityResults;
import microsoft.exchange.webservices.data.misc.availability.TimeWindow;
import microsoft.exchange.webservices.data.property.complex.Attendee;
import microsoft.exchange.webservices.data.property.complex.AttendeeCollection;
import microsoft.exchange.webservices.data.property.complex.EmailAddress;
import microsoft.exchange.webservices.data.property.complex.FolderId;
import microsoft.exchange.webservices.data.property.complex.ItemId;
import microsoft.exchange.webservices.data.property.complex.Mailbox;
import microsoft.exchange.webservices.data.property.complex.MessageBody;
import microsoft.exchange.webservices.data.property.complex.recurrence.pattern.Recurrence;
import microsoft.exchange.webservices.data.property.complex.time.TimeZoneDefinition;
import microsoft.exchange.webservices.data.search.CalendarView;
import microsoft.exchange.webservices.data.search.FindItemsResults;
import microsoft.exchange.webservices.data.search.ItemView;
import microsoft.exchange.webservices.data.search.filter.SearchFilter;
import microsoft.exchange.webservices.data.util.TimeZoneUtils;

/**
 * The connector used by the messaging connector plugin when connecting to Exchange Server.<br>
 * Implemented through the Microsoft EWS API.
 */
public class EWSConnector extends AbstractMessagingConnector
{
    /** The avalon role */
    public static final String INNER_ROLE = EWSConnector.class.getName();
    
    private UserManager _userManager;
    private WebSessionAttributeProvider _sessionAttributeProvider;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _sessionAttributeProvider = (WebSessionAttributeProvider) manager.lookup(SessionAttributeProvider.ROLE);
    }

    /**
     * Get the service of connexion to the server exchange
     * @param userIdentity The user identity
     * @return the service
     * @throws URISyntaxException if an error occurred
     */
    protected ExchangeService getService(UserIdentity userIdentity) throws URISyntaxException
    {
        if (userIdentity == null)
        {
            return null;
        }
        String url = Config.getInstance().getValue("org.ametys.plugins.exchange.url");

        String identity = Config.getInstance().getValue("org.ametys.plugins.exchange.identity");
        ExchangeService service = null;
        switch (identity)
        {
            case "impersonate":
                service = _getImpersonatedService(userIdentity);
                break;
            case "userprefs":
                service = _getServiceByUserPrefs(userIdentity);
                break;
            case "session":
                service = _getServiceBySession(userIdentity);
                break;
            default:
                break;
        }

        if (service != null)
        {
            service.setUrl(new URI(url));
        }
        return service;
    }

    @Override
    public boolean supportUserCredential(UserIdentity userIdentity)
    {
        String identity = Config.getInstance().getValue("org.ametys.plugins.exchange.identity");
        return identity.equals("userprefs");
    }

    private String _getUserPrincipalName(UserIdentity userIdentity)
    {
        String authMethod = Config.getInstance().getValue("org.ametys.plugins.exchange.authmethodews");
        if ("email".equals(authMethod))
        {
            User user = _userManager.getUser(userIdentity);
            String email = user.getEmail();
            if (StringUtils.isBlank(email))
            {
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn("The user '" + userIdentity.getLogin() + "' has no email address set, thus exchange cannot be contacted using 'email' authentication method");
                }
                return null;
            }
            return email;
        }
        else
        {
            return userIdentity.getLogin();
        }
    }

    private ExchangeService _getServiceByUserPrefs(UserIdentity userIdentity)
    {
        String userName = _getUserPrincipalName(userIdentity);
        String password = null;

        try
        {
            password = getUserPassword(userIdentity);
            
            if (userName != null && password != null)
            {
                ExchangeService service = _initService(userName, password);
                return service;
            }
            else if (password == null)
            {
                throw new MessagingConnectorException("Missing exchange password for user " + userIdentity, ExceptionType.UNAUTHORIZED);
            }
            return null;
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Unable to get exchange user password for user'" + userIdentity.getLogin() + "'", e);
            return null;
        }
    }
    
    private ExchangeService _getServiceBySession(UserIdentity userIdentity)
    {
        String userName = _getUserPrincipalName(userIdentity);
        Optional<Object> password = _sessionAttributeProvider.getSessionAttribute(FormCredentialProvider.PASSWORD_SESSION_ATTRIBUTE);
        
        if (userName != null && password.isPresent())
        {
            ExchangeService service = _initService(userName, (String) password.get());
            return service;
        }
        else if (password.isEmpty())
        {
            throw new MessagingConnectorException("Missing exchange password for user " + userIdentity
                                                + ".\n Check that you use a FormCredentialProvider with the corresponding parameter checked", ExceptionType.UNAUTHORIZED);
        }
        
        return null;
    }

    private ExchangeService _getImpersonatedService(UserIdentity userIdentity)
    {
        String userName = Config.getInstance().getValue("org.ametys.plugins.exchange.username");
        String password = Config.getInstance().getValue("org.ametys.plugins.exchange.password");
        ExchangeService service = _initService(userName, password);

        String authMethod = Config.getInstance().getValue("org.ametys.plugins.exchange.authmethodews");

        if ("email".equals(authMethod))
        {
            User user = _userManager.getUser(userIdentity);
            String email = user.getEmail();
            if (StringUtils.isBlank(email))
            {
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn("The user '" + userIdentity.getLogin() + "' has no email address set, thus exchange cannot be contacted using 'email' authentication method");
                }
                return null;
            }
            service.setImpersonatedUserId(new ImpersonatedUserId(ConnectingIdType.SmtpAddress, email));
        }
        else
        {
            service.setImpersonatedUserId(new ImpersonatedUserId(ConnectingIdType.PrincipalName, userIdentity.getLogin()));
        }
        return service;
    }

    private ExchangeService _initService(String userName, String password)
    {
        ExchangeService service = new ExchangeService(ExchangeVersion.Exchange2010_SP2);
        ExchangeCredentials credentials = new WebCredentials(userName, password);
        service.setCredentials(credentials);
        
        return service;
    }

    @Override
    protected List<CalendarEvent> internalGetEvents(UserIdentity userIdentity, int maxDays, int maxEvents) throws MessagingConnectorException
    {
        try
        {        
            List<CalendarEvent> calendar = new ArrayList<>();
            ExchangeService service = getService(userIdentity);

            if (service != null)
            {
                // The search filter to get futur or not terminated events
                CalendarFolder cf = CalendarFolder.bind(service, WellKnownFolderName.Calendar);
    
                ZonedDateTime nowZdt = ZonedDateTime.now();
                Date fromDate = DateUtils.asDate(nowZdt.withSecond(0));
                Date untilDate = DateUtils.asDate(nowZdt.withHour(0).withMinute(0).withSecond(0).plusDays(maxDays));
                
                CalendarView calendarView = new CalendarView(fromDate, untilDate);
                FindItemsResults<Appointment> findResultsEvent = cf.findAppointments(calendarView);
    
                calendarView.setMaxItemsReturned(maxEvents > 0 ? maxEvents : null);
                findResultsEvent = cf.findAppointments(calendarView);
    
                for (Appointment event : findResultsEvent.getItems())
                {
                    CalendarEvent newEvent = new CalendarEvent();
                    newEvent.setStartDate(event.getStart());
                    newEvent.setEndDate(event.getEnd());
                    newEvent.setSubject(event.getSubject());
                    newEvent.setLocation(event.getLocation());
                    calendar.add(newEvent);
                }
                
            }
            return calendar;
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, userIdentity);
            throw new MessagingConnectorException("Failed to get the events for user " + userIdentity.toString(), type, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get the events for user " + userIdentity.toString(), e);
        }
    }
    
    @Override
    protected int internalGetEventsCount(UserIdentity userIdentity, int maxDays) throws MessagingConnectorException
    {
        try
        {
            int nextEventsCount = 0;
            ExchangeService service = getService(userIdentity);
            if (service != null)
            {
                // The search filter to get futur or not terminated events
                CalendarFolder cf = CalendarFolder.bind(service, WellKnownFolderName.Calendar);
    
                ZonedDateTime nowZdt = ZonedDateTime.now();
                Date fromDate = DateUtils.asDate(nowZdt.withSecond(0));
                Date untilDate = DateUtils.asDate(nowZdt.withHour(0).withMinute(0).withSecond(0).plusDays(maxDays));
                
                CalendarView calendarView = new CalendarView(fromDate, untilDate);
                FindItemsResults<Appointment> findResultsEvent = cf.findAppointments(calendarView);
                nextEventsCount = findResultsEvent.getTotalCount();
            }
            return nextEventsCount;
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, userIdentity);
            throw new MessagingConnectorException("Failed to get the events count for user " + userIdentity.toString(), type, e);
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get the events count for user " + userIdentity.toString(), e);
        }
    }
    
    @Override
    protected List<EmailMessage> internalGetEmails(UserIdentity userIdentity, int maxEmails) throws MessagingConnectorException
    {
        try
        {
            List<EmailMessage> mailMessage = new ArrayList<>();
            
            ExchangeService service = getService(userIdentity);

            if (service != null)
            {
                // The search filter to get unread email
                SearchFilter sf = new SearchFilter.SearchFilterCollection(LogicalOperator.And, new SearchFilter.IsEqualTo(EmailMessageSchema.IsRead, false));
                ItemView view = new ItemView(maxEmails);
                FindItemsResults<Item> findResultsMail = service.findItems(WellKnownFolderName.Inbox, sf, view);
    
                List<Item> messagesReceived = findResultsMail.getItems();
                for (Item message : messagesReceived)
                {
                    message.load();
    
                    EmailMessage newMessage = new EmailMessage();
                    EmailAddress sender = ((microsoft.exchange.webservices.data.core.service.item.EmailMessage) message).getSender();
                    if (sender != null)
                    {
                        newMessage.setSender(sender.getAddress());
                    }
                    if (message.getSubject() != null)
                    {
                        newMessage.setSubject(message.getSubject());
                    }
                    if (message.getBody() != null)
                    {
                        newMessage.setSummary(html2text(message.getBody().toString()));
                    }
                    mailMessage.add(newMessage);
                }
            }
            return mailMessage;
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, userIdentity);
            throw new MessagingConnectorException("Failed to get the emails for user " + userIdentity.toString(), type, e);
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get the emails for user " + userIdentity.toString(), e);
        }
        
    }
    
    @Override
    protected int internalGetEmailsCount(UserIdentity userIdentity) throws MessagingConnectorException
    {
        try
        {
            int emailsCount = 0;
            
            ExchangeService service = getService(userIdentity);

            if (service != null)
            {
                // The search filter to get unread email
                SearchFilter sf = new SearchFilter.SearchFilterCollection(LogicalOperator.And, new SearchFilter.IsEqualTo(EmailMessageSchema.IsRead, false));
                ItemView view = new ItemView(20);
                FindItemsResults<Item> findResultsMail = service.findItems(WellKnownFolderName.Inbox, sf, view);
                
                emailsCount = findResultsMail.getTotalCount();
            }
            return emailsCount;
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, userIdentity);
            throw new MessagingConnectorException("Failed to get the emails for user " + userIdentity.toString(), type, e);
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get the emails for user " + userIdentity.toString(), e);
        }
    }

    @Override
    public boolean supportInvitation() throws MessagingConnectorException
    {
        return true;
    }

    @Override
    public boolean internalIsEventExist(String eventId, UserIdentity organiser) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                ItemId itemId = new ItemId(eventId);
                Appointment appointment = Appointment.bind(service, itemId);
                
                return appointment != null;
            }
        }
        catch (ServiceResponseException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            if (type == ExceptionType.UNKNOWN)
            {
                return false; //Exchange doesn't find the event with id 'event Id'
            }
            else
            {
                //Throw an exception if this is a known error
                throw new MessagingConnectorException("Failed to get event " + eventId + " from organiser " + organiser.toString(), type, e);
            }
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e) 
        {
            throw new MessagingConnectorException("Failed to get event " + eventId + " from organiser " + organiser.toString(), e);
        }
        
        return false;
    }
    
    @Override
    public String internalCreateEvent(String title, String description, String place, boolean isAllDay, ZonedDateTime startDate, ZonedDateTime endDate, EventRecurrenceTypeEnum recurrenceType, ZonedDateTime untilDate, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                Appointment appointment = new Appointment(service);
                
                _setDataEvent(service, appointment, title, description, place, isAllDay, DateUtils.asDate(startDate), DateUtils.asDate(endDate), recurrenceType, DateUtils.asDate(untilDate), attendees);
                
                User organiserUser = _userManager.getUser(organiser);
                Mailbox mailBox = new Mailbox(organiserUser.getEmail());
                appointment.save(new FolderId(WellKnownFolderName.Calendar, mailBox), SendInvitationsMode.SendOnlyToAll);
                
                return appointment.getId().getUniqueId();
            }
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            throw new MessagingConnectorException("Failed to create event from organiser " + organiser.toString(), type, e);
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to create event from organiser " + organiser.toString(), e);
        }
            
        return null;
    }
    
    @Override
    public void internalUpdateEvent(String eventId, String title, String description, String place, boolean isAllDay, ZonedDateTime startDate, ZonedDateTime endDate, EventRecurrenceTypeEnum recurrenceType, ZonedDateTime untilDate, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                ItemId itemId = new ItemId(eventId);
                Appointment appointment = Appointment.bind(service, itemId);

                _setDataEvent(service, appointment, title, description, place, isAllDay, DateUtils.asDate(startDate), DateUtils.asDate(endDate), recurrenceType, DateUtils.asDate(untilDate), attendees);
                
                appointment.update(ConflictResolutionMode.AlwaysOverwrite, SendInvitationsOrCancellationsMode.SendOnlyToAll);
            }
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            throw new MessagingConnectorException("Failed to update event from organiser " + organiser.toString(), type, e);
        }
        catch (MessagingConnectorException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to update event from organiser " + organiser.toString(), e);
        }
    }
    
    private void _setDataEvent(ExchangeService service, Appointment appointment, String title, String description, String place, boolean isAllDay, Date startDate, Date endDate, EventRecurrenceTypeEnum recurrenceType, Date untilDate, Map<String, Boolean> attendees) throws Exception
    {
        TimeZone defaultTimeZone = TimeZone.getDefault();
        Map<String, String> olsonTimeZoneToMsMap = TimeZoneUtils.createOlsonTimeZoneToMsMap();
        String msTimeZoneId = olsonTimeZoneToMsMap.get(defaultTimeZone.getID());

        Collection<TimeZoneDefinition> serverTimeZones = service.getServerTimeZones(Collections.singletonList(msTimeZoneId));
        TimeZoneDefinition timeZone = serverTimeZones.iterator().next();
        
        appointment.setSubject(title);
        appointment.setBody(new MessageBody(description));
        appointment.setStart(startDate);
        if (isAllDay)
        {
            Date date = Date.from(endDate.toInstant().atZone(ZoneId.systemDefault()).plusDays(1).toInstant());
            appointment.setEnd(date);
        }
        else
        {
            appointment.setEnd(endDate);
        }
        appointment.setIsAllDayEvent(isAllDay);
        appointment.setLocation(place);
        appointment.setStartTimeZone(timeZone);
        appointment.setEndTimeZone(timeZone);
        
        _setRecurrence(appointment, startDate, recurrenceType, untilDate);
        
        _setAttendees(appointment, attendees);
    }
    
    private void _setRecurrence(Appointment appointment, Date startDate, EventRecurrenceTypeEnum recurrenceType, Date untilDate) throws Exception
    {
        Recurrence recurrence = null;
        switch (recurrenceType)
        {
            case ALL_DAY:
                recurrence = new Recurrence.DailyPattern(startDate, 1);
                break;
            case ALL_WORKING_DAY:
                String workingDayAsString = Config.getInstance().getValue("org.ametys.plugins.explorer.calendar.event.working.day");
                
                List<DayOfTheWeek> days = new ArrayList<>();
                for (String idDay : StringUtils.split(workingDayAsString, ","))
                {
                    days.add(EnumUtils.getEnumList(DayOfTheWeek.class).get(Integer.parseInt(idDay) - 1));
                }
                
                recurrence = new Recurrence.WeeklyPattern(startDate, 1, days.toArray(new DayOfTheWeek[days.size()]));
                break;
            case WEEKLY:
                ZonedDateTime startWeeklyDateTime = startDate.toInstant().atZone(ZoneId.systemDefault());
                int dayOfWeekForWeekly = startWeeklyDateTime.getDayOfWeek().getValue();
                
                recurrence = new Recurrence.WeeklyPattern(startDate, 1, EnumUtils.getEnumList(DayOfTheWeek.class).get(dayOfWeekForWeekly % 7));
                break;
            case BIWEEKLY:
                ZonedDateTime startBiWeeklyDateTime = startDate.toInstant().atZone(ZoneId.systemDefault());
                int dayOfWeekForBiWeekly = startBiWeeklyDateTime.getDayOfWeek().getValue();
                
                recurrence = new Recurrence.WeeklyPattern(startDate, 2, EnumUtils.getEnumList(DayOfTheWeek.class).get(dayOfWeekForBiWeekly % 7));
                break;
            case MONTHLY:
                ZonedDateTime startMonthlyDateTime = startDate.toInstant().atZone(ZoneId.systemDefault());
                int dayOfMonth = startMonthlyDateTime.getDayOfMonth();
                
                recurrence = new Recurrence.MonthlyPattern(startDate, 1, dayOfMonth);
                break;
            case NEVER:
            default:
                //Still null
                break;
        }

        if (untilDate != null && recurrence != null)
        {
            recurrence.setEndDate(untilDate);
            appointment.setRecurrence(recurrence);
        }
    }

    @Override
    public void internalDeleteEvent(String eventId, UserIdentity organiser) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                ItemId itemId = new ItemId(eventId);
                Appointment appointment = Appointment.bind(service, itemId);
                appointment.delete(DeleteMode.MoveToDeletedItems);
            }
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            throw new MessagingConnectorException("Failed to delete event " + eventId + " with organiser " + organiser.toString(), type, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to delete event " + eventId + " with organiser " + organiser.toString(), e);
        }
            
    }

    @Override
    public Map<String, AttendeeInformation> internalGetAttendees(String eventId, UserIdentity organiser) throws MessagingConnectorException
    {
        Map<String, AttendeeInformation> attendees = new HashMap<>();
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                ItemId itemId = new ItemId(eventId);
                Appointment appointment = Appointment.bind(service, itemId);
                
                for (Attendee attendee : appointment.getRequiredAttendees())
                {
                    ResponseType responseStatus = _getResponseStatus(attendee.getResponseType());
                    AttendeeInformation attendeeInformation = new AttendeeInformation(true, responseStatus);
                    attendees.put(attendee.getAddress(), attendeeInformation);
                }
                
                for (Attendee attendee : appointment.getOptionalAttendees())
                {
                    ResponseType responseStatus = _getResponseStatus(attendee.getResponseType());
                    AttendeeInformation attendeeInformation = new AttendeeInformation(false, responseStatus);
                    attendees.put(attendee.getAddress(), attendeeInformation);
                }
            }
        }
        catch (ServiceResponseException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            if (type == ExceptionType.UNKNOWN)
            {
                return attendees; //Exchange doesn't find the event with id 'event Id'
            }
            else
            {
                throw new MessagingConnectorException("Failed to get attendees from event " + eventId + " with organiser " + organiser.toString(), type, e);
            }
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get attendees from event " + eventId + " with organiser " + organiser.toString(), e);
        }
        
        return attendees;
    }

    @Override
    public void internalSetAttendees(String eventId, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                ItemId itemId = new ItemId(eventId);
                Appointment appointment = Appointment.bind(service, itemId);
                
                _setAttendees(appointment, attendees);
                
                appointment.update(ConflictResolutionMode.AlwaysOverwrite, SendInvitationsOrCancellationsMode.SendOnlyToChanged);
            }
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            throw new MessagingConnectorException("Failed to get attendees from event " + eventId + " with organiser " + organiser.toString(), type, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get attendees from event " + eventId + " with organiser " + organiser.toString(), e);
        }
    }

    @Override
    public Map<String, FreeBusyStatus> internalGetFreeBusy(Date startDate, Date endDate, boolean isAllDay, Set<String> attendees, UserIdentity organiser) throws MessagingConnectorException
    {
        Map<String, FreeBusyStatus> attendeesMap = new HashMap<>();
        if (attendees.isEmpty())
        {
            return attendeesMap;
        }
        
        try
        {
            ExchangeService service = getService(organiser);
            if (service != null)
            {
                TimeWindow timeWindow = null;
                if (isAllDay)
                {
                    Date endDatePlus1 = Date.from(endDate.toInstant().atZone(ZoneId.systemDefault()).plusDays(1).toInstant());
                    timeWindow = new TimeWindow(startDate, endDatePlus1);
                }
                else
                {
                    Date startDateMinus1 = Date.from(startDate.toInstant().atZone(ZoneId.systemDefault()).minusDays(1).toInstant());
                    Date endDatePlus1 = Date.from(endDate.toInstant().atZone(ZoneId.systemDefault()).plusDays(1).toInstant());
                    timeWindow = new TimeWindow(startDateMinus1, endDatePlus1);
                }

                List<AttendeeInfo> attendeesInfo = new ArrayList<>();
                for (String email : attendees)
                {
                    attendeesInfo.add(new AttendeeInfo(email));
                }
                
                GetUserAvailabilityResults userAvailability = service.getUserAvailability(attendeesInfo, timeWindow, AvailabilityData.FreeBusy);
                int index = 0;
                for (AttendeeAvailability availability : userAvailability.getAttendeesAvailability())
                {
                    AttendeeInfo attendeeInfo = attendeesInfo.get(index);
                    String email = attendeeInfo.getSmtpAddress();
                    
                    FreeBusyStatus freeBusyStatus = FreeBusyStatus.Unknown;
                    if (!ServiceResult.Error.equals(availability.getResult()))
                    {
                        freeBusyStatus = FreeBusyStatus.Free;
                        for (microsoft.exchange.webservices.data.property.complex.availability.CalendarEvent calEvent : availability.getCalendarEvents())
                        {
                            if (isAllDay)
                            {
                                if (calEvent.getFreeBusyStatus().equals(LegacyFreeBusyStatus.Busy))
                                {
                                    freeBusyStatus = FreeBusyStatus.Busy;
                                }
                            }
                            else
                            {
                                if (calEvent.getFreeBusyStatus().equals(LegacyFreeBusyStatus.Busy) && startDate.before(calEvent.getEndTime()) && endDate.after(calEvent.getStartTime()))
                                {
                                    freeBusyStatus = FreeBusyStatus.Busy;
                                }
                            }
                        }
                    }
                    
                    attendeesMap.put(email, freeBusyStatus);
                    index++;
                }
            }
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, organiser);
            throw new MessagingConnectorException("Failed to get free/busy with organiser " + organiser.toString(), type, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to get free/busy with organiser " + organiser.toString(), e);
        }
        
        return attendeesMap;
    }
    
    @Override
    public boolean isUserExist(UserIdentity userIdentity) throws MessagingConnectorException
    {
        try
        {
            ExchangeService service = getService(userIdentity);
            if (service != null)
            {
                Folder.bind(service, WellKnownFolderName.Inbox);
                return true;
            }
            
            return false;
        }
        catch (ServiceRequestException e)
        {
            Throwable cause = e.getCause();
            ExceptionType type = _getExceptionType(cause, userIdentity);
            if (type == ExceptionType.UNKNOWN)
            {
                return false;
            }
            else
            {
                throw new MessagingConnectorException("Failed to know if user " + userIdentity.getLogin() + " exist in exchange", type, e);
            }
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("Failed to know if user " + userIdentity.getLogin() + " exist in exchange", e);
        }
    }
    
    private ResponseType _getResponseStatus(MeetingResponseType meetingResponseType)
    {
        switch (meetingResponseType)
        {
            case Accept:
                return ResponseType.Accept;
            case Decline:
                return ResponseType.Decline;
            case Tentative:
                return ResponseType.Maybe;
            default:
                return ResponseType.Unknown;
        }
    }
    
    private void _setAttendees(Appointment appointment, Map<String, Boolean> attendees) throws Exception
    {
        if (attendees != null)
        {
            AttendeeCollection requiredAttendees = appointment.getRequiredAttendees();
            AttendeeCollection optionalAttendees = appointment.getOptionalAttendees();
            
            requiredAttendees.clear();
            optionalAttendees.clear();
            for (String email : attendees.keySet())
            {
                boolean isMandatory = attendees.get(email);
                if (isMandatory)
                {
                    requiredAttendees.add(new Attendee(email));
                }
                else
                {
                    optionalAttendees.add(new Attendee(email));
                }
            }
        }
    }
    
    /**
     * Converts a given html String into a plain text String
     * @param html the html String that will be converted
     * @return a String plain text of the given html
     */
    protected static String html2text(String html)
    {
        return Jsoup.parse(html).text();
    }

    /**
     * Get the type of exception from the Throwable
     * @param exception exception thrown by Exchange API
     * @param userIdentity The user involved
     * @return {@link ExceptionType}
     */
    private ExceptionType _getExceptionType(Throwable exception, UserIdentity userIdentity)
    {
        ExceptionType type = ExceptionType.UNKNOWN;
        if (exception == null)
        {
            return ExceptionType.UNKNOWN;
        }

        HttpErrorException httpException = null;
        if (exception instanceof HttpErrorException)
        {
            httpException = (HttpErrorException) exception;
        }
        if (exception.getCause() instanceof HttpErrorException)
        {
            httpException = (HttpErrorException) exception.getCause();
        }

        if (httpException != null)
        {
            int httpErrorCode = httpException.getHttpErrorCode();
            if (httpErrorCode == 401)
            {
                if (!supportUserCredential(userIdentity))
                {
                    // Impersonation, so this is not a problem about the user but a configuration exception
                    type = ExceptionType.CONFIGURATION_EXCEPTION;
                }
                else
                {
                    type = ExceptionType.UNAUTHORIZED;
                }
            }
            else if (httpErrorCode == 404)
            {
                type = ExceptionType.CONFIGURATION_EXCEPTION;
            }
        }
        else if (exception.getCause() instanceof UnknownHostException)
        {
            type = ExceptionType.CONFIGURATION_EXCEPTION;
        }
        else if (exception.getCause() instanceof SocketTimeoutException)
        {
            type = ExceptionType.TIMEOUT;
        }
        return type;
    }
}
