/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.exchange;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.plugins.messagingconnector.AbstractMessagingConnector;
import org.ametys.plugins.messagingconnector.CalendarEvent;
import org.ametys.plugins.messagingconnector.EmailMessage;
import org.ametys.plugins.messagingconnector.MessagingConnectorException;
import org.ametys.runtime.config.Config;

/**
 * The connector used by the messaging connector plugin when the exchange mail server is used. 
 */
public class ExchangeConnector extends AbstractMessagingConnector
{
    private UserManager _userManager;
    
    private EWSConnector _ewsConnector;
    private GraphConnector _graphConnector;

    private List<String> _ewsUserDirectoryIds;
    private List<String> _graphUserDirectoryIds;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _ewsConnector = (EWSConnector) manager.lookup(EWSConnector.INNER_ROLE);
        _graphConnector = (GraphConnector) manager.lookup(GraphConnector.INNER_ROLE);
    }   

    @Override
    public void initialize()
    {
        super.initialize();

        List<String> allowedUserDirectoryIds = null;
        try
        {
            allowedUserDirectoryIds = _getAllowedUserDirectoryIds(List.of());
            
        }
        catch (MessagingConnectorException e)
        {
            getLogger().error("An error occured during computation of allowed user directories", e);
            return;
        }
        
        _ewsUserDirectoryIds = new ArrayList<>();
        _graphUserDirectoryIds = new ArrayList<>();
        
        Boolean ews = Config.getInstance().getValue("org.ametys.plugins.exchange.ews");
        Boolean graph = Config.getInstance().getValue("org.ametys.plugins.exchange.graph");


        if (ews)
        {
            String userDirectoryIdsAsString = Config.getInstance().getValue("org.ametys.plugins.exchange.ewsuserdirectory");
            _ewsUserDirectoryIds = _getUserDirectoryIds(userDirectoryIdsAsString, allowedUserDirectoryIds);
        }

        if (graph)
        {
            String userDirectoryIdsAsString = Config.getInstance().getValue("org.ametys.plugins.exchange.graphuserdirectory");
            _graphUserDirectoryIds = _getUserDirectoryIds(userDirectoryIdsAsString, allowedUserDirectoryIds);
        }
        
        if (graph && ews)
        {
            List<String> allowedExchangeUserDirectoryIds = _getAllowedUserDirectoryIds(_ewsUserDirectoryIds);
            List<String> allowedGraphUserDirectoryIds = _getAllowedUserDirectoryIds(_graphUserDirectoryIds);
            
            if (CollectionUtils.containsAny(allowedExchangeUserDirectoryIds, allowedGraphUserDirectoryIds))
            {
                throw new IllegalStateException("The directory ids defined in Azure application and Exchange Server share same user directory");
            }
        }
    }
    
    private List<String> _getUserDirectoryIds(String userDirectoryIdsAsString, List<String> allowedUserDirectoryIds)
    {
        List<String> ids = new ArrayList<>();
        
        if (StringUtils.isNotBlank(userDirectoryIdsAsString))
        {
            String[] userDirectoryIds = StringUtils.split(userDirectoryIdsAsString, ",");
            
            List<String> wrongUserDirectoryIds = new ArrayList<>();
            for (String userDirectoryId : userDirectoryIds)
            {
                String userDirectoryIdTrimed = StringUtils.trim(userDirectoryId);
                if (!allowedUserDirectoryIds.contains(userDirectoryIdTrimed))
                {
                    wrongUserDirectoryIds.add(userDirectoryIdTrimed);
                }
                else
                {
                    ids.add(userDirectoryIdTrimed);
                }
            }
            
            if (!wrongUserDirectoryIds.isEmpty())
            {
                throw new IllegalStateException("The following user directory ids defined in the Exchange configuration do not exist : " + wrongUserDirectoryIds);
            }
        }
        
        return ids;
    }
    
    private List<String> _getAllowedUserDirectoryIds(List<String> userDirectoryIds)
    {
        if (userDirectoryIds.isEmpty())
        {
            List<String> allowedUserDirectoryIds = getAllowedPopulationIds()
                    .stream()
                    .map(populationId -> _userPopulationDAO.getUserPopulation(populationId))
                    .map(UserPopulation::getUserDirectories)
                    .flatMap(List::stream)
                    .map(UserDirectory::getId)
                    .collect(Collectors.toList());
            return allowedUserDirectoryIds;
        }
        else
        {
            return userDirectoryIds;
        }
    }

    private ConnectorType _getConnectorTypeByUserIdentity(UserIdentity userIdentity)
    {
        User user = _userManager.getUser(userIdentity);
        UserDirectory userDirectory = user.getUserDirectory();
        String userDirectoryId = userDirectory.getId();
        return _getConnectorTypeByUserDirectory(userDirectoryId);
    }
    
    private ConnectorType _getConnectorTypeByUserDirectory(String userDirectoryId)
    {
        Boolean exchange = Config.getInstance().getValue("org.ametys.plugins.exchange.ews");
        Boolean graph = Config.getInstance().getValue("org.ametys.plugins.exchange.graph");
        
        if (!exchange && !graph)
        {
            return ConnectorType.NO_CONNECTOR_CONFIGURED;
        }
        else if (exchange && !graph)
        {
            return ConnectorType.EXCHANGE_CONNECTOR;
        }
        else if (!exchange && graph)
        {
            return ConnectorType.GRAPH_CONNECTOR;
        }
        else if (exchange && graph)
        {
            boolean idIsInExchangeList = _getAllowedUserDirectoryIds(_ewsUserDirectoryIds).contains(userDirectoryId);
            boolean idIsInGraphList = _getAllowedUserDirectoryIds(_graphUserDirectoryIds).contains(userDirectoryId);
            
            if (idIsInExchangeList && !idIsInGraphList)
            {
                return ConnectorType.EXCHANGE_CONNECTOR;
            }
            else if (!idIsInExchangeList && idIsInGraphList)
            {
                return ConnectorType.GRAPH_CONNECTOR;
            }
        }

        return ConnectorType.NO_CONNECTOR_ASSIGNED;
    }
    
    /**
     * Rights profiles
     */
    private enum ConnectorType
    {
        /** Graph connector */
        GRAPH_CONNECTOR,
        /** Exchange connector */
        EXCHANGE_CONNECTOR,
        /** No connector configured */
        NO_CONNECTOR_CONFIGURED,
        /** No connector assigned */
        NO_CONNECTOR_ASSIGNED;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
    }
    
    @Override
    protected List<CalendarEvent> internalGetEvents(UserIdentity userIdentity, int maxDays, int maxEvents) throws MessagingConnectorException
    {
        switch (_getConnectorTypeByUserIdentity(userIdentity))
        {
            case GRAPH_CONNECTOR:
                return _graphConnector.internalGetEvents(userIdentity, maxDays, maxEvents);
            case EXCHANGE_CONNECTOR:
                return _ewsConnector.internalGetEvents(userIdentity, maxDays, maxEvents);
            default:
                throw new MessagingConnectorException("Cannot get events for user " + userIdentity + ": user directory is not supported by this messaging connector", MessagingConnectorException.ExceptionType.CONFIGURATION_EXCEPTION);
        }
    }
    
    @Override
    protected int internalGetEventsCount(UserIdentity userIdentity, int maxDays) throws MessagingConnectorException
    {
        switch (_getConnectorTypeByUserIdentity(userIdentity))
        {
            case GRAPH_CONNECTOR:
                return _graphConnector.internalGetEventsCount(userIdentity, maxDays);
            case EXCHANGE_CONNECTOR:
                return _ewsConnector.internalGetEventsCount(userIdentity, maxDays);
            default:
                throw new MessagingConnectorException("Cannot get events count for user " + userIdentity + ": user directory is not supported by this messaging connector", MessagingConnectorException.ExceptionType.CONFIGURATION_EXCEPTION);
        }
    }
    
    @Override
    protected List<EmailMessage> internalGetEmails(UserIdentity userIdentity, int maxEmails) throws MessagingConnectorException
    {
        switch (_getConnectorTypeByUserIdentity(userIdentity))
        {
            case GRAPH_CONNECTOR:
                return _graphConnector.internalGetEmails(userIdentity, maxEmails);
            case EXCHANGE_CONNECTOR:
                return _ewsConnector.internalGetEmails(userIdentity, maxEmails);
            default:
                throw new MessagingConnectorException("Cannot get emails for user " + userIdentity + ": user directory is not supported by this messaging connector", MessagingConnectorException.ExceptionType.CONFIGURATION_EXCEPTION);
        }
    }
    
    @Override
    protected int internalGetEmailsCount(UserIdentity userIdentity) throws MessagingConnectorException
    {
        switch (_getConnectorTypeByUserIdentity(userIdentity))
        {
            case GRAPH_CONNECTOR:
                return _graphConnector.internalGetEmailsCount(userIdentity);
            case EXCHANGE_CONNECTOR:
                return _ewsConnector.internalGetEmailsCount(userIdentity);
            default:
                throw new MessagingConnectorException("Cannot get emails count for user " + userIdentity + ": user directory is not supported by this messaging connector", MessagingConnectorException.ExceptionType.CONFIGURATION_EXCEPTION);
        }
    }
    
    @Override
    public boolean supportUserCredential(UserIdentity userIdentity)
    {
        switch (_getConnectorTypeByUserIdentity(userIdentity))
        {
            case GRAPH_CONNECTOR:
                return _graphConnector.supportUserCredential(userIdentity);
            case EXCHANGE_CONNECTOR:
                return _ewsConnector.supportUserCredential(userIdentity);
            default:
                throw new MessagingConnectorException("Cannot get support for credential for user " + userIdentity + ": user directory is not supported by this messaging connector", MessagingConnectorException.ExceptionType.CONFIGURATION_EXCEPTION);
        }
    }
}
