/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.exchange;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.extrausermgt.graph.GraphClientProvider;
import org.ametys.plugins.extrausermgt.graph.GraphClientProvider.GraphClientException;
import org.ametys.plugins.messagingconnector.AbstractMessagingConnector;
import org.ametys.plugins.messagingconnector.CalendarEvent;
import org.ametys.plugins.messagingconnector.EmailMessage;
import org.ametys.plugins.messagingconnector.MessagingConnectorException;
import org.ametys.plugins.messagingconnector.MessagingConnectorException.ExceptionType;

import com.microsoft.graph.core.tasks.PageIterator;
import com.microsoft.graph.models.Event;
import com.microsoft.graph.models.EventCollectionResponse;
import com.microsoft.graph.models.Message;
import com.microsoft.graph.models.MessageCollectionResponse;
import com.microsoft.graph.users.item.UserItemRequestBuilder;

/**
 * The connector used by the messaging connector plugin when connecting to Exchange Online.<br>
 * Implemented through the Microsoft Graph API.
 */
public class GraphConnector extends AbstractMessagingConnector
{
    /** The avalon role */
    public static final String INNER_ROLE = GraphConnector.class.getName();
    
    /** The Graph client provider */
    protected GraphClientProvider _graphClientProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _graphClientProvider = (GraphClientProvider) manager.lookup(GraphClientProvider.ROLE);
    }
    
    private List<Event> _getEvents(UserIdentity userIdentity, int maxDays)
    {
        try
        {
            UserItemRequestBuilder userRequestBuilder = _graphClientProvider.getUserRequestBuilder(userIdentity);
            
            ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.SECONDS);
            
            String start = now.format(DateTimeFormatter.ISO_DATE_TIME);
            String end = now.plusDays(maxDays).format(DateTimeFormatter.ISO_DATE_TIME);
            
            List<Event> result = new ArrayList<>();

            EventCollectionResponse eventCollectionResponse = userRequestBuilder.calendar().calendarView().get(requestConfiguration -> {
                requestConfiguration.queryParameters.startDateTime = start;
                requestConfiguration.queryParameters.endDateTime = end;
                requestConfiguration.queryParameters.orderby = new String[] {"start/datetime"};
                requestConfiguration.queryParameters.select = new String[] {"subject", "start", "end", "location"};
            });
            
            new PageIterator.Builder<Event, EventCollectionResponse>()
                            .client(_graphClientProvider.getGraphClient())
                            .collectionPage(eventCollectionResponse)
                            .collectionPageFactory(EventCollectionResponse::createFromDiscriminatorValue)
                            .processPageItemCallback(event -> {
                                result.add(event);
                                return true;
                            })
                            .build()
                            .iterate();
            
            return result;
        }
        catch (GraphClientException e)
        {
            throw new MessagingConnectorException("Failed to retrieve a Graph client for user " + userIdentity + ". "
                    + "The user could not be linked to a Entra user", ExceptionType.UNAUTHORIZED, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("An error occured while contacting the graph endpoint for user " + userIdentity, ExceptionType.UNKNOWN, e);
        }
    }
    
    @Override
    protected List<CalendarEvent> internalGetEvents(UserIdentity userIdentity, int maxDays, int maxEvents) throws MessagingConnectorException
    {
        List<Event> events = _getEvents(userIdentity, maxDays);
        
        Map<Long, CalendarEvent> calendarEvents = new TreeMap<>();
        for (Event event : events)
        {
            LocalDateTime ldt = LocalDateTime.parse(event.getStart().getDateTime(), DateTimeFormatter.ISO_DATE_TIME);
            ZonedDateTime zdt = ldt.atZone(ZoneId.of(event.getStart().getTimeZone()));
            zdt = zdt.withZoneSameInstant(ZoneId.systemDefault());
            
            long longStart = zdt.toEpochSecond();
            Date startDate = DateUtils.asDate(zdt);
            
            ldt = LocalDateTime.parse(event.getEnd().getDateTime(), DateTimeFormatter.ISO_DATE_TIME);
            zdt = ldt.atZone(ZoneId.of(event.getEnd().getTimeZone()));
            zdt = zdt.withZoneSameInstant(ZoneId.systemDefault());
            
            Date endDate = DateUtils.asDate(zdt);
            
            CalendarEvent newEvent = new CalendarEvent();
            newEvent.setStartDate(startDate);
            newEvent.setEndDate(endDate);
            newEvent.setSubject(event.getSubject());
            newEvent.setLocation(event.getLocation().getDisplayName());
            calendarEvents.put(longStart, newEvent);
        }
        
        return calendarEvents.entrySet().stream().limit(maxEvents).map(e -> e.getValue()).collect(Collectors.toList());
    }

    @Override
    protected int internalGetEventsCount(UserIdentity userIdentity, int maxDays) throws MessagingConnectorException
    {
        return _getEvents(userIdentity, maxDays).size();
    }

    @Override
    protected List<EmailMessage> internalGetEmails(UserIdentity userIdentity, int maxEmails) throws MessagingConnectorException
    {
        try
        {
            UserItemRequestBuilder userRequestBuilder = _graphClientProvider.getUserRequestBuilder(userIdentity);
            
            MessageCollectionResponse messageCollectionResponse = userRequestBuilder.mailFolders().byMailFolderId("inbox").messages().get(requestConfiguration -> {
                requestConfiguration.queryParameters.top = Math.min(maxEmails, 999);
                requestConfiguration.queryParameters.select = new String[] {"sender", "subject", "bodyPreview"};
                requestConfiguration.queryParameters.filter = "isRead eq false";
            });
            
            List<EmailMessage> result = new ArrayList<>();
            
            new PageIterator.Builder<Message, MessageCollectionResponse>()
                            .client(_graphClientProvider.getGraphClient())
                            .collectionPage(messageCollectionResponse)
                            .collectionPageFactory(MessageCollectionResponse::createFromDiscriminatorValue)
                            .processPageItemCallback(message -> {
                                result.add(new EmailMessage(message.getSubject(), message.getSender().getEmailAddress().getAddress(), message.getBodyPreview()));
                                return result.size() < maxEmails;
                            })
                            .build()
                            .iterate();
            
            return result;
        }
        catch (GraphClientException e)
        {
            throw new MessagingConnectorException("Failed to retrieve a Graph client for user " + userIdentity + ". "
                    + "The user could not be linked to a Entra user", ExceptionType.UNAUTHORIZED, e);
        }
        catch (Exception e)
        {
            throw new MessagingConnectorException("An error occured while contacting the graph endpoint for user " + userIdentity, ExceptionType.UNKNOWN, e);
        }
    }

    @Override
    protected int internalGetEmailsCount(UserIdentity userIdentity) throws MessagingConnectorException
    {
        try
        {
            UserItemRequestBuilder userRequestBuilder = _graphClientProvider.getUserRequestBuilder(userIdentity);
            
            return userRequestBuilder.mailFolders().byMailFolderId("inbox").get().getUnreadItemCount();
        }
        catch (GraphClientException e)
        {
            throw new MessagingConnectorException("Failed to retrieve a Graph client for user " + userIdentity + ". "
                    + "The user could not be linked to a Entra user", ExceptionType.UNAUTHORIZED, e);
        }
    }
}
