/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data;

import java.util.Collection;
import java.util.Map;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * The DataSource Factory is responsible for building DataSource and Query objects.
 * It handles one or several DataSource/Query types.
 * @param <Q> The query
 * @param <R> The query result
 */
public interface DataSourceFactory<Q extends Query, R extends QueryResult>
{
    
    /**
     * Get the types that the factory can build.
     * @return the handled types as a Collection.
     */
    Collection<DataSourceType> getHandledTypes();
    
    /**
     * The configuration parameters needed to build a Query of the specified type.
     * @param type the type of the Query.
     * @return the configuration parameters as a Collection.
     */
    Collection<String> getQueryConfigurationParameters(String type);
    
    /**
     * Build a query with the specified information.
     * @param id the Query ID.
     * @param type the Query type.
     * @param name the Query name.
     * @param description the Query description.
     * @param resultType the Query result type (single/multiple).
     * @param dataSourceId the id of data source
     * @param additionalConfiguration additional query configuration parameter values as a Map&lt;parameterName, parameterValue&gt;.
     * @return the query.
     * @throws DataInclusionException if an error occurs trying to create the Query.
     */
    Q buildQuery(String id, String type, String name, String description, ResultType resultType, String dataSourceId, Map<String, String> additionalConfiguration) throws DataInclusionException;
    
    /**
     * Execute the query with the specified parameter values.
     * @param query The query
     * @param parameterValues the parameter values as a Map (name -&gt; value).
     * @return the query result.
     * @throws DataInclusionException if an error occurs while executing the query
     */
    R execute(Q query, Map<String, String> parameterValues) throws DataInclusionException;
    
    /**
     * Execute the query with the specified parameter values.
     * @param query The query
     * @param parameterValues the parameter values as a Map (name -&gt; value).
     * @param offset The start index of search
     * @param limit The max number of result to return
     * @return the query result.
     * @throws DataInclusionException if an error occurs while executing the query
     */
    R execute(Q query, Map<String, String> parameterValues, int offset, int limit) throws DataInclusionException;
}
