/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data;

import java.util.Map;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * Provider for connections and queries.
 */
public interface QueryDao
{
    
    /** The avalon role name. */
    public static final String ROLE = QueryDao.class.getName();
    
    /**
     * Get all the queries of a site.
     * @param siteName the site name.
     * @return the queries as a Map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Map<String, Query> getQueries(String siteName) throws DataInclusionException;
    
    /**
     * Get all the queries of a site.
     * @param siteName the site name.
     * @param type the query type
     * @return the queries as a Map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Map<String, Query> getQueries(String siteName, DataSourceType type) throws DataInclusionException;
    
    /**
     * Get all the queries of a site.
     * @param siteName the site name.
     * @param dataSourceId the id of the data source
     * @return the queries as a Map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId) throws DataInclusionException;
    
    /**
     * Get all the queries of a site of a specific result type
     * @param siteName the site name.
     * @param dataSourceId the id of the data source
     * @param resultType Filter on result type
     * @return the queries as a Map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId, ResultType resultType) throws DataInclusionException;
    
    /**
     * Get all the queries of a site of a specific result type
     * @param siteName the site name.
     * @param dataSourceId the id of the data source
     * @param dataSourceType the query type
     * @param resultType Filter on result type
     * @return the queries as a Map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId, DataSourceType dataSourceType, ResultType resultType) throws DataInclusionException;
    
    /**
     * Get a query from its id.
     * @param siteName the site name.
     * @param id the query id.
     * @return the Query.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    Query getQuery(String siteName, String id) throws DataInclusionException;
    
    /**
     * Get the query properties.
     * @param id The query id
     * @param siteName The site name
     * @return The properties of the query in a result map.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    @Callable(rights = {"Datainclusion_Right_Content_InsertQuery", "Datainclusion_Right_Manage"}, context = "/cms")
    Map<String, Object> getQueryProperties(String id, String siteName) throws DataInclusionException;
    
    /**
     * Adds a query.
     * @param siteName The site name
     * @param parameters The params needed to create the query
     * @return The id of the created query,its parent id and its type, or an error
     * @throws Exception if an error occurs when adding the query
     */
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    Map<String, String> addQuery(String siteName, Map<String, Object> parameters) throws Exception;
    
    /**
     * Updates a query.
     * @param siteName The site name
     * @param parameters The params needed to update the query
     * @return The id of the updated query and its type, or an error
     * @throws Exception if an error occurs when updating the query
     */
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    Map<String, String> updateQuery(String siteName, Map<String, Object> parameters) throws Exception;
    
    /**
     * Deletes a query.
     * @param siteName the site name
     * @param id The id of the query to delete
     * @return The id of the deleted query, or an error
     * @throws Exception if an error occurs when deleting the query
     */
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    Map<String, String> deleteQuery(String siteName, String id) throws Exception;
    
    /**
     * Add a query and set its id.
     * @param siteName the site name
     * @param query the query to add. The id member doesn't need to be specified.
     * @return id of the newly created data source.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    String addQuery(String siteName, Query query) throws DataInclusionException;
    
    /**
     * Update a query. Update the query specified by its id member with all the data.
     * @param siteName the site name
     * @param query the query to update.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    void updateQuery(String siteName, Query query) throws DataInclusionException;
    
    /**
     * Remove a query.
     * @param siteName the site name
     * @param id the query id.
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    void removeQuery(String siteName, String id) throws DataInclusionException;
    
}
