/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.jcr;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.externaldata.cache.ObservationConstants;
import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.DataSourceFactory;
import org.ametys.plugins.externaldata.data.DataSourceFactoryExtensionPoint;
import org.ametys.plugins.externaldata.data.Query;
import org.ametys.plugins.externaldata.data.Query.ResultType;
import org.ametys.plugins.externaldata.data.QueryDao;
import org.ametys.plugins.externaldata.data.QueryResult;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.web.repository.site.SiteManager;

/**
 * JCR implementation of the Query DAO.
 * This class manages DataSource and Query objects, storing them in the JCR repository.
 */
public class JcrQueryDao extends AbstractLogEnabled implements QueryDao, Serviceable, Component
{
    
    /** JCR relative path to root node. */
    public static final String ROOT_REPO = AmetysObjectResolver.ROOT_REPO;
    
    /** Plugins root node name. */
    public static final String PLUGINS_NODE = "ametys-internal:plugins";
    
    /** Plugin root node name. */
    public static final String PLUGIN_NODE = "external-data";
    
    /** Data sources node name. */
    public static final String DATASOURCES_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":datasources";
    
    /** Queries node name. */
    public static final String QUERIES_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":queries";
    
    /** "Name" property name. */
    public static final String PROPERTY_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":name";
    
    /** "Description" property name. */
    public static final String PROPERTY_DESCRIPTION = RepositoryConstants.NAMESPACE_PREFIX + ":description";
    
    /** "Type" property name. */
    public static final String PROPERTY_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":type";
    
    /** "Type" property name. */
    public static final String PROPERTY_DATASOURCE = RepositoryConstants.NAMESPACE_PREFIX + ":dataSourceId";
    
    /** Configuration properties prefix. */
    public static final String PROPERTY_CONF_PREFIX = RepositoryConstants.NAMESPACE_PREFIX + ":conf-";
    
    /** Query "query string" property name. */
    public static final String QUERY_PROPERTY_QUERYSTRING = RepositoryConstants.NAMESPACE_PREFIX + ":queryString";
    
    /** Query "parameters" property name. */
    public static final String QUERY_PROPERTY_PARAMETERS = RepositoryConstants.NAMESPACE_PREFIX + ":parameters";
    
    /** Query "resultType" property name. */
    public static final String QUERY_PROPERTY_RESULTTYPE = RepositoryConstants.NAMESPACE_PREFIX + ":resultType";
    
    /** The Data Source factory extension point. */
    protected DataSourceFactoryExtensionPoint _dataSourceFactoryEP;
    
    /** The JCR repository. */
    protected Repository _repository;
    
    /** The Site manager. */
    protected SiteManager _siteManager;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The observation manager */
    protected ObservationManager _observationManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _dataSourceFactoryEP = (DataSourceFactoryExtensionPoint) serviceManager.lookup(DataSourceFactoryExtensionPoint.ROLE);
        _repository = (Repository) serviceManager.lookup(AbstractRepository.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
    }
    
    @Override
    public Map<String, Query> getQueries(String siteName) throws DataInclusionException
    {
        try
        {
            Map<String, Query> queries = new HashMap<>();
            
            Node queriesNode = _getQueriesNode(siteName);
            
            NodeIterator queryNodes = queriesNode.getNodes();
            while (queryNodes.hasNext())
            {
                Node node = queryNodes.nextNode();
                
                // Extract a Query object from the node and add it to the map.
                Query query = _extractQuery(node);
                queries.put(query.getId(), query);
            }
            
            return queries;
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error getting the data sources", e);
        }
    }
    
    @Override
    public Map<String, Query> getQueries(String siteName, DataSourceType type) throws DataInclusionException
    {
        return getDataSourceQueries(siteName, null, type, null);
    }
    
    @Override
    public Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId) throws DataInclusionException
    {
        return getDataSourceQueries(siteName, dataSourceId, null);
    }
    
    @Override
    public Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId, ResultType resultType) throws DataInclusionException
    {
        return getDataSourceQueries(siteName, dataSourceId, null, resultType);
    }
    
    @Override
    public Map<String, Query> getDataSourceQueries(String siteName, String dataSourceId, DataSourceType dataSourceType, ResultType resultType) throws DataInclusionException
    {
        try
        {
            // TODO Should be a made with a query instead of filtering.
            Map<String, Query> queries = new HashMap<>();
            
            Node queriesNode = _getQueriesNode(siteName);
            
            NodeIterator queryNodes = queriesNode.getNodes();
            while (queryNodes.hasNext())
            {
                Node node = queryNodes.nextNode();
                
                // Extract a Query object from the node and add it to the map.
                Query query = _extractQuery(node);
                if (_matchQuery(query, dataSourceId, dataSourceType, resultType))
                {
                    queries.put(query.getId(), query);
                }
            }
            
            return queries;
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error getting the data sources", e);
        }
    }
    
    private boolean _matchQuery (Query query, String dataSourceId, DataSourceType dsType, ResultType resultType)
    {
        if ((dataSourceId == null || query.getDataSourceId().equals(dataSourceId))
                && (dsType == null || query.getType().equals(dsType))
                && (resultType == null || resultType.equals(query.getResultType())))
        {
            return true;
        }
        
        return false;
    }
    
    @Override
    public Query getQuery(String siteName, String id) throws DataInclusionException
    {
        try
        {
            Query query = null;
            
            Node queriesNode = _getQueriesNode(siteName);
            
            if (queriesNode.hasNode(id))
            {
                Node node = queriesNode.getNode(id);
                
                // Extract a Query object from the node.
                query = _extractQuery(node);
            }
            
            return query;
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error getting the data source of id " + id, e);
        }
    }
    
    @Override
    @Callable(rights = {"Datainclusion_Right_Content_InsertQuery", "Datainclusion_Right_Manage"}, context = "/cms")
    public Map<String, Object> getQueryProperties(String id, String siteName) throws DataInclusionException
    {
        Map<String, Object> infos = new HashMap<>();
        
        Query query = getQuery(siteName, id);
        if (query != null)
        {
            infos.put("id", query.getId());
            infos.put("name", query.getName());
            infos.put("description", query.getDescription());
            infos.put("type", query.getType());
            infos.put("resultType", query.getResultType().name());
            
            Map<String, String> additionalConf = query.getAdditionalConfiguration();
            for (String confName : additionalConf.keySet())
            {
                String value = additionalConf.get(confName);
                if (value != null)
                {
                    infos.put(confName, value);
                }
            }
            
            infos.put("parameters", query.getParameters());
            
            infos.put("dataSourceId", query.getDataSourceId());
        }
        
        return infos;
    }
    
    @Override
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    public Map<String, String> addQuery(String siteName, Map<String, Object> parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        String type = (String) parameters.get("type");
        String name = (String) parameters.get("name");
        String description = (String) parameters.get("description");
        String resultTypeStr = (String) parameters.get("resultType");
        ResultType resultType = _getResultType(resultTypeStr, ResultType.MULTIPLE);
        String dataSourceId = (String) parameters.get("dataSourceId");
        
        DataSourceFactory<Query, QueryResult> factory = _dataSourceFactoryEP.getFactory(DataSourceType.valueOf(type));
        if (factory == null)
        {
            getLogger().error("Impossible to get a factory to handle the type : " + type);
            result.put("error", "unknown-type");
            return result;
        }
        Collection<String> params = factory.getQueryConfigurationParameters(type);
        
        Map<String, String> additionalConfiguration = new HashMap<>();
        for (String paramName : params)
        {
            String value = (String) parameters.get(paramName);
            if (value != null)
            {
                additionalConfiguration.put(paramName, value);
            }
        }
        
        try
        {
            Query query = factory.buildQuery(null, type, name, description, resultType, dataSourceId, additionalConfiguration);
            
            String id = addQuery(siteName, query);
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Query created : got id " + id);
            }
            
            result.put("id", id);
            result.put("parentId", dataSourceId);
            result.put("type", type);
        }
        catch (DataInclusionException e)
        {
            getLogger().error("Error saving the query", e);
            result.put("error", "add-query-error");
        }
        
        return result;
    }
    
    @Override
    public String addQuery(String siteName, Query query) throws DataInclusionException
    {
        try
        {
            Node queriesNode = _getQueriesNode(siteName);
            
            String name = query.getName();
            if (StringUtils.isBlank(name))
            {
                throw new DataInclusionException("Query name can't be blank.");
            }
            
            String nodeName = NameHelper.filterName(name);
            String notExistingNodeName = _getNotExistingNodeName(queriesNode, nodeName);
            
            // TODO Create a query JCR node type.
            Node node = queriesNode.addNode(notExistingNodeName);
            String id = node.getName();
            node.addMixin(JcrConstants.MIX_REFERENCEABLE);
            
            _fillQueryNode(query, node);
            
            queriesNode.getSession().save();
            
            return id;
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error adding a Query", e);
        }
    }
    
    @Override
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    public Map<String, String> updateQuery(String siteName, Map<String, Object> parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        String id = (String) parameters.get("id");
        String type = (String) parameters.get("type");
        String name = (String) parameters.get("name");
        String description = (String) parameters.get("description");
        String resultTypeStr = (String) parameters.get("resultType");
        ResultType resultType = _getResultType(resultTypeStr, ResultType.MULTIPLE);
        
        DataSourceFactory<Query, QueryResult> factory = _dataSourceFactoryEP.getFactory(DataSourceType.valueOf(type));
        Collection<String> params = factory.getQueryConfigurationParameters(type);
        
        Map<String, String> additionalConfiguration = new HashMap<>();
        for (String paramName : params)
        {
            String value = (String) parameters.get(paramName);
            if (value != null)
            {
                additionalConfiguration.put(paramName, value);
            }
        }
        
        try
        {
            Query query = factory.buildQuery(id, type, name, description, resultType, null, additionalConfiguration);
            
            updateQuery(siteName, query);
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Query updated : id " + query.getId());
            }
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_QUERY_ID, query.getId());
            _observationManager.notify(new Event(ObservationConstants.EVENT_QUERY_UPDATED, _currentUserProvider.getUser(), eventParams));
            
            result.put("id", id);
            result.put("type", type);
        }
        catch (DataInclusionException e)
        {
            getLogger().error("Error updating the query", e);
            result.put("error", "update-query-error");
        }
        
        return result;
    }
    
    @Override
    public void updateQuery(String siteName, Query query) throws DataInclusionException
    {
        String id = query.getId();
        
        try
        {
            Node queriesNode = _getQueriesNode(siteName);
            
            if (!queriesNode.hasNode(id))
            {
                throw new DataInclusionException("No query exists with id " + id);
            }
            
            Node node = queriesNode.getNode(id);
            
            String name = query.getName();
            if (StringUtils.isBlank(name))
            {
                throw new DataInclusionException("Name can't be blank.");
            }
            
            _fillQueryNode(query, node);
            
            node.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error updating the Data Source of id " + id, e);
        }
    }
    
    @Override
    @Callable(rights = "Datainclusion_Right_Manage", context = "/cms")
    public Map<String, String> deleteQuery(String siteName, String id) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        try
        {
            removeQuery(siteName, id);
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Query id " + id + " deleted.");
            }
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_QUERY_ID, id);
            _observationManager.notify(new Event(ObservationConstants.EVENT_QUERY_DELETED, _currentUserProvider.getUser(), eventParams));
            
            result.put("id", id);
        }
        catch (DataInclusionException e)
        {
            getLogger().error("Error deleting the query", e);
            result.put("error", "delete-query-error");
        }
        
        return result;
    }
    
    @Override
    public void removeQuery(String siteName, String id) throws DataInclusionException
    {
        try
        {
            Node queriesNode = _getQueriesNode(siteName);
            
            if (!queriesNode.hasNode(id))
            {
                throw new DataInclusionException("No data source exists with id " + id);
            }
            
            queriesNode.getNode(id).remove();
            
            queriesNode.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new DataInclusionException("Error updating the Data Source of id " + id, e);
        }
    }
    
    /**
     * Create a data source from a node.
     * @param node the data source node.
     * @return the data source.
     * @throws RepositoryException if an error occurs when exploring the repository
     * @throws DataInclusionException if an error occurs while manipulating the data sources
     */
    protected Query _extractQuery(Node node) throws RepositoryException, DataInclusionException
    {
        Query query = null;
        
        String id = node.getName();
        String type = _getSingleProperty(node, PROPERTY_TYPE, "");
        String name = _getSingleProperty(node, PROPERTY_NAME, "");
        String description = _getSingleProperty(node, PROPERTY_DESCRIPTION, "");
        ResultType resultType = _getResultType(node, ResultType.MULTIPLE);
        String dataSourceId = _getSingleProperty(node, PROPERTY_DATASOURCE, "");
        Map<String, String> additionalConf = _getAdditionalConf(node);
        
        DataSourceFactory<Query, QueryResult> factory = _dataSourceFactoryEP.getFactory(DataSourceType.valueOf(type));
        
        if (factory != null)
        {
            query = factory.buildQuery(id, type, name, description, resultType, dataSourceId, additionalConf);
        }
        else
        {
            throw new DataInclusionException("Unknown query type.");
        }
        
        return query;
    }
    
    /**
     * Get a single property value.
     * @param node the JCR node.
     * @param propertyName the name of the property to get.
     * @param defaultValue the default value if the property does not exist.
     * @return the single property value.
     * @throws RepositoryException if a repository error occurs.
     */
    protected String _getSingleProperty(Node node, String propertyName, String defaultValue) throws RepositoryException
    {
        String value = defaultValue;
        
        if (node.hasProperty(propertyName))
        {
            value = node.getProperty(propertyName).getString();
        }
        
        return value;
    }
    
    /**
     * Get a a result type
     * @param value the value
     * @param defaultValue the default result type
     * @return the result type for the value
     * @throws RepositoryException if an error occurs when exploring the repository
     */
    protected ResultType _getResultType(String value, ResultType defaultValue) throws RepositoryException
    {
        ResultType result = defaultValue;
        try
        {
            result = ResultType.valueOf(value);
        }
        catch (Exception e)
        {
            // Ignore
        }
        
        return result;
    }
    
    /**
     * Get a single property value.
     * @param node the node
     * @param defaultValue the default result type
     * @return the result type for the value
     * @throws RepositoryException if an error occurs when exploring the repository
     */
    protected ResultType _getResultType(Node node, ResultType defaultValue) throws RepositoryException
    {
        ResultType result = defaultValue;
        if (node.hasProperty(QUERY_PROPERTY_RESULTTYPE))
        {
            String value = node.getProperty(QUERY_PROPERTY_RESULTTYPE).getString();
            try
            {
                result = ResultType.valueOf(value);
            }
            catch (Exception e)
            {
                // Ignore
            }
        }
        
        return result;
    }
    
    /**
     * Get the values of a string array property.
     * @param node the node.
     * @param propertyName the name of the property to get.
     * @return the values.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Collection<String> _getMultipleProperty(Node node, String propertyName) throws RepositoryException
    {
        List<String> values = new ArrayList<>();
        
        if (node.hasProperty(propertyName))
        {
            Value[] propertyValues = node.getProperty(propertyName).getValues();
            for (Value value : propertyValues)
            {
                values.add(value.getString());
            }
        }
        
        return values;
    }
    
    /**
     * Get additional configuration from properties.
     * @param node the node
     * @return the additional configuration as a Map.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Map<String, String> _getAdditionalConf(Node node) throws RepositoryException
    {
        Map<String, String> values = new HashMap<>();
        
        PropertyIterator propertyIt = node.getProperties(PROPERTY_CONF_PREFIX + "*");
        while (propertyIt.hasNext())
        {
            Property property = propertyIt.nextProperty();
            String propName = property.getName();
            String name = propName.substring(PROPERTY_CONF_PREFIX.length(), propName.length());
            String value = property.getString();
            
            values.put(name, value);
        }
        
        return values;
    }
    
    /**
     * Store the query properties into the JCR node.
     * @param query the querys
     * @param node the query node
     * @throws RepositoryException if a repository error occurs.
     */
    protected void _fillQueryNode(Query query, Node node) throws RepositoryException
    {
        node.setProperty(PROPERTY_NAME, query.getName());
        node.setProperty(PROPERTY_DESCRIPTION, query.getDescription());
        node.setProperty(PROPERTY_TYPE, query.getType().name());
        node.setProperty(QUERY_PROPERTY_RESULTTYPE, query.getResultType().name());
        if (query.getDataSourceId() != null)
        {
            node.setProperty(PROPERTY_DATASOURCE, query.getDataSourceId());
        }
        
        Map<String, String> additionalConf = query.getAdditionalConfiguration();
        for (String confName : additionalConf.keySet())
        {
            String value = additionalConf.get(confName);
            node.setProperty(PROPERTY_CONF_PREFIX + confName, value);
        }
    }
    
    /**
     * Get a name for a node which doesn't already exist in this node.
     * @param container the container node.
     * @param baseName the base wanted node name.
     * @return the name, free to be taken.
     * @throws RepositoryException if a repository error occurs.
     */
    protected String _getNotExistingNodeName(Node container, String baseName) throws RepositoryException
    {
        String name = baseName;
        
        int index = 2;
        while (container.hasNode(name))
        {
            name = baseName + index;
            index++;
        }
        
        return name;
    }
    
    /**
     * Get the plugin root node in a site storage space.
     * @param siteName the site name.
     * @return the plugin root node.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Node _getPluginNode(String siteName) throws RepositoryException
    {
        
        Node pluginNode;
        Session session = null;
        try
        {
            session = _repository.login();
            ModifiableTraversableAmetysObject rootPlugins = _siteManager.getSite(siteName).getRootPlugins();
            Node pluginsNode = ((JCRAmetysObject) rootPlugins).getNode();
            
            if (pluginsNode.hasNode(PLUGIN_NODE))
            {
                pluginNode = pluginsNode.getNode(PLUGIN_NODE);
            }
            else
            {
                pluginNode = pluginsNode.addNode(PLUGIN_NODE);
                pluginsNode.getSession().save();
            }
            
            return pluginNode;
        }
        catch (PathNotFoundException e)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("Unable to get site plugins node for site " + siteName, e);
        }
        catch (RepositoryException e)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occured while getting site plugins node for site " + siteName, e);
        }
    }
    
    /**
     * Get the queries root node.
     * @param siteName the name of the site
     * @return the queries root node.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Node _getQueriesNode(String siteName) throws RepositoryException
    {
        Node node;
        try
        {
            Node pluginNode = _getPluginNode(siteName);
            
            if (pluginNode.hasNode(QUERIES_NODE))
            {
                node = pluginNode.getNode(QUERIES_NODE);
            }
            else
            {
                node = pluginNode.addNode(QUERIES_NODE);
                pluginNode.getSession().save();
            }
            
            return node;
        }
        catch (PathNotFoundException e)
        {
            throw new AmetysRepositoryException("Unable to get queries node because it doesn't exist.", e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occured while getting queries node.", e);
        }
    }
    
}
