/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.ldap;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.core.util.ldap.AbstractLDAPConnector;
import org.ametys.core.util.ldap.ScopeEnumerator;
import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.DataSourceFactory;
import org.ametys.plugins.externaldata.data.Query.ResultType;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * LDAP Data Source factory.
 */
public class LdapDataSourceFactory extends AbstractLDAPConnector implements DataSourceFactory<LdapQuery, LdapQueryResult>, PluginAware
{
    /** The relative DN for users */
    public static final String QUERY_CONFIGURATION_RELATIVE_DN = "relativeDN";
    
    /** The search scope of the query */
    public static final String QUERY_CONFIGURATION_SCOPE = "scope";
    
    /** The LDAp attributes to return */
    public static final String QUERY_CONFIGURATION_ATTRIBUTES = "attributes";
    
    /** Constraint query parameter. */
    public static final String QUERY_CONFIGURATION_CONSTRAINT = "constraint";
    
    /** Query configuration parameters. */
    public static final List<String> QUERY_CONFIGURATION_PARAMETERS = Arrays.asList(
        QUERY_CONFIGURATION_RELATIVE_DN,
        QUERY_CONFIGURATION_SCOPE,
        QUERY_CONFIGURATION_ATTRIBUTES,
        QUERY_CONFIGURATION_CONSTRAINT);
    
    private String _id;
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    @Override
    public Collection<DataSourceType> getHandledTypes()
    {
        return Collections.singleton(DataSourceType.LDAP);
    }
    
    
    @Override
    public Collection<String> getQueryConfigurationParameters(String type)
    {
        return QUERY_CONFIGURATION_PARAMETERS;
    }
    
    
    @Override
    public LdapQuery buildQuery(String id, String type, String name, String description, ResultType resultType, String dataSourceId, Map<String, String> additionalConfiguration) throws DataInclusionException
    {
        String relativeDN = StringUtils.defaultString(additionalConfiguration.get(QUERY_CONFIGURATION_RELATIVE_DN));
        String attributes = StringUtils.defaultString(additionalConfiguration.get(QUERY_CONFIGURATION_ATTRIBUTES));
        String constraint = StringUtils.defaultString(additionalConfiguration.get(QUERY_CONFIGURATION_CONSTRAINT));
        String scope = StringUtils.defaultString(additionalConfiguration.get(QUERY_CONFIGURATION_SCOPE));
        
//        if (StringUtils.isBlank(attributes))
//        {
//            throw new DataInclusionException("Impossible to build the LDAP query : returning attributes must be provided");
//        }
        
        LdapQuery query = new LdapQuery();
        
        query.setId(id);
        query.setFactory(this._id);
        query.setName(name);
        query.setDescription(description);
        query.setResultType(resultType);
        query.setDataSourceId(dataSourceId);
        
        query.setRelativeDN(relativeDN);
        query.setAttributes(attributes);
        query.setConstraint(constraint);
        query.setScope(scope);
        
        return query;
    }
    
    @Override
    public LdapQueryResult execute(LdapQuery query, Map<String, String> parameterValues) throws DataInclusionException
    {
        return execute(query, parameterValues, 0, Integer.MAX_VALUE);
    }
    
    public LdapQueryResult execute(LdapQuery query, Map<String, String> parameterValues, int offset, int limit) throws DataInclusionException
    {
        List<SearchResult> results = null;
        try
        {
            String dataSourceId = query.getDataSourceId();
            _delayedInitialize(dataSourceId);
            
            Map<String, String> attributeMap = query.getAttributesAsMap();
            
            int paramCount = query.getParameters().size();
            Object[] values = new Object[paramCount];
            
            // Replace all the parameters by "{x}" placeholders and fill the parameter values array.
            int paramIndex = 0;
            String constraint = query.getConstraint();
            for (String paramName : query.getParameters().keySet())
            {
                if (parameterValues.containsKey(paramName))
                {
                    constraint = constraint.replaceAll("\\$\\{" + paramName + "(\\[[^\\]]*\\])?\\}", "{" + paramIndex + "}");
                    values[paramIndex] = parameterValues.get(paramName);
                    paramIndex++;
                }
                else
                {
                    String regexp = "\\([^()=,\\s]*=[^$()]*\\$\\{" + paramName + "(\\[[^\\]]*\\])?\\}[^\\)]*\\)";
                    constraint = constraint.replaceAll(regexp, "");
                }
            }
            
            // Search filter.
            StringBuilder filter = new StringBuilder();
            // filter.append("(&").append(_dataSource.getFilter());
            if (StringUtils.isNotBlank(constraint))
            {
                filter.append(constraint);
            }
            
            List<String> columnLabels = new ArrayList<>(attributeMap.keySet());
            
            // Create the controls : the attributes to return are the values of the map.
            SearchControls controls = _getSearchControls(attributeMap.values(), query.getScope());
            
            // Execute the search.
            results = _search(query.getRelativeDN(), filter.toString(), values, controls, offset, limit);
            
            // Extract and return the results.
            LdapQueryResult queryResult = new LdapQueryResult(results, columnLabels, attributeMap);
            
            return queryResult;
        }
        catch (Exception e)
        {
            throw new DataInclusionException("Unable to execute the LDAP query.", e);
        }
    }
    
    /**
     * Get the LDAP search controls.
     * @param attributes the attributes to return.
     * @param scopeStr the scope as a String.
     * @return the search controls.
     * @throws DataInclusionException If an error occurred
     */
    protected SearchControls _getSearchControls(Collection<String> attributes, String scopeStr) throws DataInclusionException
    {
        // Paramètres de recherche
        SearchControls controls = new SearchControls();
        
        String[] attrArray = attributes.toArray(new String[attributes.size()]);
        
        controls.setReturningAttributes(attrArray);
        
        controls.setSearchScope(_getScope(scopeStr));
        
        return controls;
    }
    
    /**
     * Get the scope as an integer (handlable by the SearchControls) from the scope string.
     * @param scopeStr the scope string.
     * @return the scope as an integer.
     * @throws DataInclusionException If an error occurred
     */
    protected int _getScope(String scopeStr) throws DataInclusionException
    {
        try
        {
            return ScopeEnumerator.parseScope(scopeStr);
        }
        catch (IllegalArgumentException e)
        {
            throw new DataInclusionException("Unable to parse scope", e);
        }
    }
}
