/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.ldap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchResult;

import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.QueryResult;
import org.ametys.plugins.externaldata.data.QueryResultRow;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * LDAP query result.
 */
public class LdapQueryResult implements QueryResult
{
    
    /** The result type. */
    protected ResultType _resultType;
    
    /** The result rows. */
    protected List<QueryResultRow> _rows;
    
    /** The column names (ordered) */
    protected List<String> _columnNames;
    
    /** The attribute labels. */
    protected Map<String, String> _attributes;

    /**
     * Constructs a query result around a ResultSet.
     * @param results the list of search results
     * @param columnNames the names of the columns
     * @param attributes the attributes
     */
    public LdapQueryResult(List<SearchResult> results, List<String> columnNames, Map<String, String> attributes)
    {
        try
        {
            _columnNames = columnNames;
            _attributes = attributes;
            _extractResults(results, columnNames, attributes);
        }
        catch (NamingException e)
        {
            throw new RuntimeException(e);
        }
    }
    
    @Override
    public ResultType getType()
    {
        return _resultType;
    }
    
    /**
     * Set the result type.
     * @param resultType the result type.
     */
    public void setType(ResultType resultType)
    {
        this._resultType = resultType;
    }
    
    @Override
    public int getSize()
    {
        return _rows.size();
    }
    
    @Override
    public Collection<String> getColumnNames() throws DataInclusionException
    {
        return _columnNames;
    }
    
    @Override
    public Iterator<QueryResultRow> iterator()
    {
        return _rows.iterator();
    }
    
    @Override
    public void close()
    {
        // Does nothing.
    }
    
    /**
     * Extract the column names and results from a LDAP naming enumeration.
     * @param results the LDAP naming enumeration.
     * @param columnNames the names of the columns
     * @param attributes the attributes
     * @throws NamingException if something goes wrong when manipulating the context
     */
    protected void _extractResults(List<SearchResult> results, List<String> columnNames, Map<String, String> attributes) throws NamingException
    {
        // Extract the results.
        _rows = new ArrayList<>();
        
        for (SearchResult result : results)
        {
            // Get attributes
            Attributes attrs = result.getAttributes();
            
            LdapQueryResultRow row = new LdapQueryResultRow();
            
            for (String colName : _columnNames)
            {
                String attrId = attributes.get(colName);
                Attribute attrValue = attrs.get(attrId);
                if (attrValue != null)
                {
                    row.put(colName, (String) attrValue.get());
                }
            }
            
            _rows.add(row);
        }
    }
    
}
