/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.transformation;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.AbstractEnhancementHandler;
import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.DataSourceFactory;
import org.ametys.plugins.externaldata.data.DataSourceFactoryExtensionPoint;
import org.ametys.plugins.externaldata.data.Query;
import org.ametys.plugins.externaldata.data.Query.ResultType;
import org.ametys.plugins.externaldata.data.QueryDao;
import org.ametys.plugins.externaldata.data.QueryResult;
import org.ametys.plugins.externaldata.data.QueryResultRow;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.content.WebContent;

/**
 * DataInclusion enhancement handler : transform a query tag into its results.
 */
public class DataInclusionEnhancementHandler extends AbstractEnhancementHandler implements Component, Serviceable, Contextualizable, PluginAware
{
    /** The Avalon role. */
    public static final String ROLE = DataInclusionEnhancementHandler.class.getName();
    
    private static final String _QUERY_TAG = "dataquery";
    
    private static final String _ID_ATTR = "id";
    
    private static final String _PARAMETER_PREFIX = "param-";
    
    /** The Query DAO. */
    protected QueryDao _queryDao;
    
    /** The avalon context. */
    protected Context _context;
    
    /** The plugin name. */
    protected String _pluginName;
    
    /** The logger */
    protected Logger _logger;
    
    private RenderingContextHandler _renderingContextHandler;
    private URIPrefixHandler _prefixHandler;

    private DataSourceFactoryExtensionPoint _dataSourceFactoryEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _queryDao = (QueryDao) serviceManager.lookup(QueryDao.ROLE);
        _dataSourceFactoryEP = (DataSourceFactoryExtensionPoint) serviceManager.lookup(DataSourceFactoryExtensionPoint.ROLE);
        _renderingContextHandler = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
        _prefixHandler = (URIPrefixHandler) serviceManager.lookup(URIPrefixHandler.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    @Override
    public void enableLogging(final Logger logger)
    {
        _logger = logger;
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException
    {
        if (_QUERY_TAG.equals(localName))
        {
            Request request = ContextHelper.getRequest(_context);
            WebContent content = (WebContent) request.getAttribute(Content.class.getName());
            String contentTitle = content.getTitle();
            String id = atts.getValue(_ID_ATTR);
            
            String siteName = (String) ContextHelper.getRequest(_context).getAttribute("site");
            if (siteName == null)
            {
                siteName = content.getSiteName();
            }
            
            try
            {
                if (StringUtils.isNotBlank(id))
                {
                    Query query = _queryDao.getQuery(siteName, id);
                    if (query != null)
                    {
                        Map<String, String> parameters = getParameters(query, atts);

                        String factoryId = query.getFactory();
                        DataSourceFactory<Query, QueryResult> dsFactory = _dataSourceFactoryEP.getExtension(factoryId);
                        QueryResult result = dsFactory.execute(query, parameters);
                        
                        if (result != null)
                        {
                            if (query.getResultType().equals(ResultType.MULTIPLE))
                            {
                                _saxResult(query, result);
                            }
                            else
                            {
                                _saxSingleResult(query, result);
                            }
                        }
                    }
                    else
                    {
                        _logger.error("The query of id '" + id + "' is included in the content of title '" + contentTitle + "' but was deleted.");
                        
                        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
                        if (currentContext == RenderingContext.BACK)
                        {
                            String iconPath = _prefixHandler.getUriPrefix() + "/plugins/" + _pluginName + "/resources/img/content/edition/data-error.png";
                            saxError(new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_ERROR_TITLE"), new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_ERROR_NONEXISTING_QUERY"), iconPath);
                        }
                    }
                }
                else
                {
                    _logger.error("Query ID must be provided.");
                    RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
                    if (currentContext == RenderingContext.BACK)
                    {
                        String iconPath = _prefixHandler.getUriPrefix() + "/plugins/" + _pluginName + "/resources/img/content/edition/data-error.png";
                        saxError(new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_ERROR_TITLE"), new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_ERROR_NO_ID"), iconPath);
                    }
                }
            }
            catch (DataInclusionException e)
            {
                _logger.error("Error executing the query " + id, e);
                RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
                if (currentContext == RenderingContext.BACK)
                {
                    String iconPath = _prefixHandler.getUriPrefix() + "/plugins/" + _pluginName + "/resources/img/content/edition/data-error.png";
                    saxError(new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_ERROR_TITLE"), new I18nizableText("plugin." + _pluginName, "PLUGINS_EXTERNAL_DATA_QUERY_EXECUTION_ERROR"), iconPath);
                }
            }
        }
        else
        {
            super.startElement(uri, localName, qName, atts);
        }
    }
    
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException
    {
        if (!_QUERY_TAG.equals(localName))
        {
            super.endElement(uri, localName, qName);
        }
    }
    
    /**
     * Sax a query result.
     * @param query the query.
     * @param result the result to generate.
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxResult(Query query, QueryResult result) throws SAXException
    {
        String queryId = query.getId();
        try
        {
            // Others enhancement handlers should not touch the generated HTML
            _contentHandler.processingInstruction("ametys-unmodifiable", "start");
            
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("class", "data " + queryId);
            XMLUtils.startElement(_contentHandler, "table", atts);
            Collection<String> colNames = result.getColumnNames();
            
            if (!colNames.isEmpty())
            {
                XMLUtils.startElement(_contentHandler, "tr");
                for (String columnName : colNames)
                {
                    XMLUtils.createElement(_contentHandler, "th", columnName);
                }
                XMLUtils.endElement(_contentHandler, "tr");
            }
            
            int index = 0;
            for (QueryResultRow row : result)
            {
                AttributesImpl attr = new AttributesImpl();
                if (index % 2 != 0)
                {
                    attr.addAttribute("", "class", "class", "CDATA", "odd");
                }
                else
                {
                    attr.addAttribute("", "class", "class", "CDATA", "even");
                }
                XMLUtils.startElement(_contentHandler, "tr", attr);
                
                for (String columnName : colNames)
                {
                    String value = row.get(columnName);
                    XMLUtils.createElement(_contentHandler, "td", StringUtils.defaultString(value));
                }
                XMLUtils.endElement(_contentHandler, "tr");
                index++;
            }
            
            XMLUtils.endElement(_contentHandler, "table");
            
            _contentHandler.processingInstruction("ametys-unmodifiable", "end");
        }
        catch (DataInclusionException e)
        {
            String message = "Unable to get the query results (query ID : " + queryId + ")";
            _logger.error(message, e);
            throw new SAXException(message, e);
        }
        finally
        {
            result.close();
        }
    }
    
    /**
     * Sax a single query result.
     * @param query the query
     * @param result the query's result
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxSingleResult(Query query, QueryResult result) throws SAXException
    {
        String queryId = query.getId();
        try
        {
            Collection<String> colNames = result.getColumnNames();
            if (colNames.size() > 0)
            {
                String colName = colNames.iterator().next();
                
                if (result.iterator().hasNext())
                {
                    QueryResultRow row = result.iterator().next();
                    
                    String value = row.get(colName);
                    
                    if (StringUtils.isNotBlank(value))
                    {
                        AttributesImpl atts = new AttributesImpl();
                        atts.addCDATAAttribute("class", "query " + queryId);
                        
                        XMLUtils.createElement(_contentHandler, "span", atts, value);
                    }
                }
            }
        }
        catch (DataInclusionException e)
        {
            String message = "Unable to get the query results (query ID : " + queryId + ")";
            _logger.error(message, e);
            throw new SAXException(message, e);
        }
        finally
        {
            result.close();
        }
    }
    
    /**
     * Get the query parameter values from the query node attributes.
     * @param query the query.
     * @param atts the query node attributes.
     * @return the parameters as a Map of parameter name -&gt; value.
     */
    protected Map<String, String> getParameters(Query query, Attributes atts)
    {
        Map<String, String> params = new HashMap<>();
        
        for (String param : query.getParameters().keySet())
        {
            String value = atts.getValue(_PARAMETER_PREFIX + param.toLowerCase());
            if (StringUtils.isNotBlank(value))
            {
                params.put(param, value);
            }
        }
        
        return params;
    }
    
}
