/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.datasource.DataSourceClientInteraction;
import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.plugins.externaldata.data.DataSourceFactoryExtensionPoint;
import org.ametys.plugins.externaldata.data.Query;
import org.ametys.plugins.externaldata.data.QueryDao;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * SAX events for datasources types, datasources and queries
 *
 */
public class GetDataSourceQueriesAction extends ServiceableAction
{
    /** The manager handling the data sources */
    protected DataSourceClientInteraction _dataSourceClientInteraction;
    /** The manager for data source factories */
    protected DataSourceFactoryExtensionPoint _dataSourceFactoryEP;
    /** The Query DAO. */
    protected QueryDao _queryDao;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _dataSourceClientInteraction = (DataSourceClientInteraction) smanager.lookup(DataSourceClientInteraction.ROLE);
        _dataSourceFactoryEP = (DataSourceFactoryExtensionPoint) smanager.lookup(DataSourceFactoryExtensionPoint.ROLE);
        _queryDao = (QueryDao) smanager.lookup(QueryDao.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String siteName = (String) jsParameters.get("siteName");
        String nodeId = (String) jsParameters.get("node");
        String nodeType = (String) jsParameters.get("nodeType");
        String resultTypeFilter = (String) jsParameters.get("resultType");
        
        ResultType resultType = null;
        if (StringUtils.isNotEmpty(resultTypeFilter))
        {
            resultType = ResultType.valueOf(resultTypeFilter);
        }
        
        @SuppressWarnings("unchecked")
        List<String> dataSourceTypeFilter = (List<String>) jsParameters.get("dataSourceType");
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        
        if ("root".equals(nodeType) || StringUtils.isBlank(nodeType))
        {
            if (dataSourceTypeFilter != null && !dataSourceTypeFilter.isEmpty())
            {
                for (String type : dataSourceTypeFilter)
                {
                    nodes.add(dataSourceTypeToJSON(DataSourceType.valueOf(type)));
                }
            }
            else
            {
                Collection<DataSourceType> allTypes = _dataSourceFactoryEP.getAllTypes();
                for (DataSourceType type : allTypes)
                {
                    nodes.add(dataSourceTypeToJSON(type));
                }
            }
        }
        else if ("dataSourceType".equals(nodeType))
        {
            DataSourceType dataSourceType = DataSourceType.valueOf(nodeId);
            
            List<Map<String, Object>> dataSources = _dataSourceClientInteraction.getDataSources(dataSourceType, false, false, false, null);
            for (Map<String, Object> dataSource : dataSources)
            {
                Collection<Query> queries = _queryDao.getDataSourceQueries(siteName, (String) dataSource.get("id"), resultType).values();
                dataSource.put("leaf", queries.isEmpty());
                dataSource.put("type", "datasource");
                dataSource.put("dataSourceType", dataSourceType.name());
                nodes.add(dataSource);
            }
        }
        else if ("datasource".equals(nodeType))
        {
            for (Query query : _queryDao.getDataSourceQueries(siteName, nodeId, null, resultType).values())
            {
                nodes.add(queryToJSON(query));
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("children", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    /**
     * Get the JSON representation of a type of data source
     * @param dataSourceType The data source type ID
     * @return The datasource type as JSON
     * @throws Exception If an error occurred
     */
    protected Map<String, Object> dataSourceTypeToJSON(DataSourceType dataSourceType) throws Exception
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", dataSourceType.name());
        infos.put("type", "dataSourceType");
        infos.put("dataSourceType", dataSourceType.name());
        infos.put("name", dataSourceType.name());
        
        List<Map<String, Object>> dataSources = _dataSourceClientInteraction.getDataSources(dataSourceType, false, false, false, null);
        infos.put("leaf", dataSources.isEmpty());
        
        return infos;
    }
    
    /**
     * Gets query's properties to JSON format
     * @param query The query
     * @return The query's properties
     */
    protected Map<String, Object> queryToJSON(Query query)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", query.getId());
        infos.put("type", "query");
        infos.put("dataSourceType", query.getType());
        infos.put("name", query.getName());
        infos.put("description", query.getDescription());
        infos.put("resultType", query.getResultType().name());
        infos.put("leaf", true);
        
        Map<String, String> additionalConf = query.getAdditionalConfiguration();
        for (String confName : additionalConf.keySet())
        {
            String value = additionalConf.get(confName);
            infos.put(confName, value);
        }
        
        infos.put("parameters", query.getParameters());
        
        return infos;
    }

}
