/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.cas;

import org.jasig.cas.client.util.CommonUtils;
import org.jasig.cas.client.util.XmlUtils;
import org.jasig.cas.client.validation.Assertion;
import org.jasig.cas.client.validation.Cas20ProxyTicketValidator;
import org.jasig.cas.client.validation.TicketValidationException;

/**
 * Ametys implementation of the TicketValidator that will validate Service Tickets in compliance with the CAS 2.
 */
public class AmetysCas20ProxyTicketValidator extends Cas20ProxyTicketValidator
{
    /** The proxy granting ticket */
    protected String _proxyGrantingTicket;

    /**
     * Constructs an instance of the CAS 2.0 Service Ticket Validator with the supplied
     * CAS server url prefix.
     *
     * @param casServerUrlPrefix the CAS Server URL prefix.
     */
    public AmetysCas20ProxyTicketValidator(final String casServerUrlPrefix)
    {
        super(casServerUrlPrefix);
        _proxyGrantingTicket = null;
    }
    
    @Override
    protected void customParseResponse(final String response, final Assertion assertion) throws TicketValidationException
    {
        super.customParseResponse(response, assertion);
        
        final String proxyGrantingTicketIou = XmlUtils.getTextForElement(response, "proxyGrantingTicket");

        if (CommonUtils.isNotBlank(proxyGrantingTicketIou))
        {
            _proxyGrantingTicket = getProxyGrantingTicketStorage().retrieve(proxyGrantingTicketIou);
            logger.debug("proxyGrantingTicketIou was found in response ({}), the associated proxyGrantingTicket is '{}'", proxyGrantingTicketIou, _proxyGrantingTicket);
        }
        else
        {
            logger.debug("While parsing response, the proxyGrantingTicketIou was not found.");
        }
    }
    
    /**
     * Gets the proxy granting ticket of the current request
     * @return The proxy granting ticket of the current request
     */
    public String getProxyGrantingTicket()
    {
        return _proxyGrantingTicket;
    }
}
