/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.oauth;

import java.io.IOException;
import java.net.URI;
import java.time.ZonedDateTime;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.StringTokenizer;

import org.ametys.core.util.DateUtils;
import org.ametys.core.util.SessionAttributeProvider;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.workspaces.extrausermgt.authentication.oauth.OAuthCallbackAction;

import com.nimbusds.oauth2.sdk.AccessTokenResponse;
import com.nimbusds.oauth2.sdk.AuthorizationGrant;
import com.nimbusds.oauth2.sdk.AuthorizationRequest;
import com.nimbusds.oauth2.sdk.AuthorizationRequest.Builder;
import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.RefreshTokenGrant;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.TokenRequest;
import com.nimbusds.oauth2.sdk.TokenResponse;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientSecretBasic;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.RefreshToken;
import com.nimbusds.oauth2.sdk.token.Tokens;

import net.minidev.json.JSONObject;
import net.minidev.json.JSONValue;

/**
 * OAuth provider definition to interact with a Nextcloud server.
 * 
 * This could be a default OAuth provider definition except for the configuration
 * part and the handling of the user id provided by Nextcloud as a custom parameter
 * in the token response.
 */
public class DefaultOauthProvider extends AbstractLogEnabled implements OAuthProvider, Configurable, Serviceable, Contextualizable
{
    /** OAuth state session attribute */
    public static final String OAUTH_STATE_SESSION_ATTRIBUTE = "oauth.state";
    /** OAuth redirect URI to use after a successful token request */
    public static final String OAUTH_REDIRECT_URI_SESSION_ATTRIBUTE = "oauth.redirect.uri";
    /** Oauth access token session attribute */
    public static final String OAUTH_ACCESS_TOKEN_SESSION_ATTRIBUTE = "oauth.access.token";
    /** Oauth access token expiration date session attribute */
    public static final String OAUTH_ACCESS_TOKEN_EXPIRATION_DATE_SESSION_ATTRIBUTE = "oauth.access.token.expiration.date";
    /** Oauth refresh token session attribute */
    public static final String OAUTH_REFRESH_TOKEN_SESSION_ATTRIBUTE = "oauth.refresh.token";
    /** Oauth custom parameter session attribute */
    public static final String OAUTH_CUSTOM_PARAMETER = "oauth.custom.parameter";
    private static final String __OAUTH_AUTHORIZATION_CALLBACK = "/_extra-user-management/oauth-callback";
    
    /** The provider id */
    protected String _id;
    /** the oauth client id */
    protected ClientID _clientID;
    /** the authentication to use when requesting a token */
    protected ClientAuthentication _auth;
    /** the authorization endpoint URI */
    protected URI _authorizationEnpoint;
    /** the token endpoint URI */
    protected URI _tokenEndpointURI;
    /** the scope for the token */
    protected Scope _scope;
    /** the list of custom parameters returned with the token that must be stored for later use*/
    protected Set<String> _customParameters;
    private SessionAttributeProvider _sessionAttributeProvider;
    
    private Set<State> _knownState = new HashSet<>();
    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sessionAttributeProvider = (SessionAttributeProvider) manager.lookup(SessionAttributeProvider.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        // Provider id
        _id = configuration.getAttribute("id");
        
        // authentication
        _clientID = new ClientID(_getConfigValue(configuration.getChild("clientId")));
        _auth = new ClientSecretBasic(_clientID, new Secret(_getConfigValue(configuration.getChild("secret"))));
        
        // endpoints
        String baseURL = _getConfigValue(configuration.getChild("baseURL"));
        if (StringUtils.isNotEmpty(baseURL))
        {
            _authorizationEnpoint = URI.create(baseURL + _getConfigValue(configuration.getChild("authorizationEndpointPath")));
            _tokenEndpointURI = URI.create(baseURL + _getConfigValue(configuration.getChild("tokenEndpointPath")));
        }
        else
        {
            _authorizationEnpoint = URI.create(_getConfigValue(configuration.getChild("authorizationEndpointURI")));
            _tokenEndpointURI = URI.create(_getConfigValue(configuration.getChild("tokenEndpointURI")));
        }
        
        // scope
        _scope = new Scope();
        String scopes = _getConfigValue(configuration.getChild("scopes"));
        for (String scope : StringTokenizer.getCSVInstance(scopes).getTokenList())
        {
            _scope.add(scope);
        }
        
        // token response custom parameters
        String customParams = _getConfigValue(configuration.getChild("customParams"));
        _customParameters = new HashSet<>(StringTokenizer.getCSVInstance(customParams).getTokenList());
    }
    
    /**
     * Get the value of a configuration element either by retrieving the associated value in
     * the application config or directly the configuration element value
     * @param cfg a configuration element. Can not be {@code null}
     * @return the value or null if the value is not present
     */
    protected String _getConfigValue(Configuration cfg)
    {
        if (cfg.getAttributeAsBoolean("config", false))
        {
            // get the config value associated or null if it doesn't exist
            return Config.getInstance().getValue(cfg.getValue(""));
        }
        else
        {
            return cfg.getValue(null);
        }
    }
    
    public ClientID getClientID()
    {
        return _clientID;
    }
    
    public URI getAuthorizationEndpointURI()
    {
        return _authorizationEnpoint;
    }
    
    public URI getTokenEndpointURI()
    {
        return _tokenEndpointURI;
    }
    
    public String getId()
    {
        return _id;
    }

    public ClientAuthentication getClientAuthentication()
    {
        return _auth;
    }

    public Scope getScope()
    {
        return _scope;
    }
    
    public Set<String> getCustomParametersName()
    {
        return _customParameters;
    }
    
    public boolean isKnownState(State state)
    {
        // actually removes it, a state should not be used multiple times
        return _knownState.remove(state);
    }
    
    public Optional<AccessToken> getStoredAccessToken()
    {
        // Get stored attribute
        Optional<AccessToken> accessToken = _sessionAttributeProvider.getSessionAttribute(OAUTH_ACCESS_TOKEN_SESSION_ATTRIBUTE + "$" + _id)
            // try to parse it or ignore it
            .map(s -> {
                try
                {
                    return AccessToken.parse((JSONObject) JSONValue.parseWithException((String) s));
                }
                catch (ParseException | net.minidev.json.parser.ParseException e)
                {
                    getLogger().warn("Failed to parse the stored access token for provider {}. The token is ignored", _id, e);
                    return null;
                }
            });
        
        if (accessToken.isEmpty())
        {
            return accessToken;
        }
        
        // Check if the token is still valid
        Optional<ZonedDateTime> expirationDate = _sessionAttributeProvider.getSessionAttribute(OAUTH_ACCESS_TOKEN_EXPIRATION_DATE_SESSION_ATTRIBUTE + "$" + _id)
                .map(str -> DateUtils.parseZonedDateTime((String) str));
        
        if (expirationDate.isEmpty())
        {
            // There is a stored token but no expiration date. This should never happens. Ignore all.
            getLogger().warn("A token is stored in session for provider {} but the token has no expiration date or an invalid one. The token is ignored.", _id);
            return Optional.empty();
        }
        
        if (ZonedDateTime.now().isBefore(expirationDate.get()))
        {
            return accessToken;
        }
        
        // The token is expired. We try to get a new one silently with the refresh token
        return _refreshToken();
    }

    private Optional<AccessToken> _refreshToken()
    {
        Optional<Object> tokenAttribute = _sessionAttributeProvider.getSessionAttribute(OAUTH_REFRESH_TOKEN_SESSION_ATTRIBUTE + "$" + _id);
        Optional<RefreshToken> refreshToken = tokenAttribute.map(str -> {
            try
            {
                return RefreshToken.parse((JSONObject) JSONValue.parseWithException((String) str));
            }
            catch (ParseException | net.minidev.json.parser.ParseException e)
            {
                getLogger().warn("Failed to parse the stored refresh token for provider {}. The token is ignored", _id, e);
                return null;
            }
        });
        
        if (refreshToken.isPresent())
        {
            AuthorizationGrant refreshTokenGrant = new RefreshTokenGrant(refreshToken.get());
            
            try
            {
                return Optional.of(requestAccessToken(refreshTokenGrant));
            }
            catch (AccessDeniedException | IOException e)
            {
                getLogger().warn("Failed to refresh access token for provider {}", _id, e);
                return Optional.empty();
            }
        }
        return Optional.empty();
    }
    
    @SuppressWarnings("unchecked")
    public <T> Optional<T> getStoredCustomParameter(String parameter)
    {
        Optional<Object> sessionAttribute = _sessionAttributeProvider.getSessionAttribute(OAUTH_CUSTOM_PARAMETER + "$" + _id + "#" + parameter);
        return sessionAttribute.map(str -> (T) JSONValue.parse((String) str));
    }
    
    public Optional<AccessToken> getAccessToken(Redirector redirector) throws ProcessingException, IOException
    {
        Optional<AccessToken> accessToken = getStoredAccessToken();
        
        if (accessToken.isPresent())
        {
            return accessToken;
        }
        
        AuthorizationRequest authRequest = buildAuthorizationCodeRequest();
        if (!redirector.hasRedirected())
        {
            redirector.redirect(false, authRequest.toURI().toString());
        }
        return Optional.empty();
    }
    
    /**
     * Build an authorization request based on the provide information.
     * 
     * The request will use a {@code ResponseType#CODE} for the response type.
     * 
     * @return an authorization request
     */
    protected AuthorizationRequest buildAuthorizationCodeRequest()
    {
        // Save the link between state and provider to retrieve provider during callback
        State state = _generateState();
        Builder authorizationRequestBuilder = new AuthorizationRequest.Builder(ResponseType.CODE, _clientID)
                .endpointURI(_authorizationEnpoint)
                .redirectionURI(_getRedirectUri())
                .state(state);
        if (!_scope.isEmpty())
        {
            authorizationRequestBuilder.scope(_scope);
        }
        return authorizationRequestBuilder.build();
    }
    
    /**
     * Generate a state for the given provider.
     * The state will be stored in the provider to be able to retrieve the provider responding
     * to a authorize request being processed in {@link OAuthCallbackAction}.
     * We also store the state in session to unsure that the response is linked to the current session.
     * 
     * @return the newly generated state
     */
    protected State _generateState()
    {
        State state = new State();
        while (!_knownState.add(state))
        {
            // if the state already existed, generate a new one
            state = new State();
        }
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession();
        // only one state is required by session as you can't go through multiple authorization process at the same time
        session.setAttribute(OAUTH_STATE_SESSION_ATTRIBUTE, state);
        return state;
    }

    private URI _getRedirectUri()
    {
        // Before returning the standard redirect URI we store the current
        // request URI in session. This way, the standard redirect will be
        // able to redirect to the current request making it transparent to
        // the user.
        Request request = ContextHelper.getRequest(_context);
        _storeCurrentRequestUriInSession(request);
        return _buildAbsoluteURI(request, __OAUTH_AUTHORIZATION_CALLBACK);
    }

    private void _storeCurrentRequestUriInSession(Request request)
    {
        // creation of the actual redirect URI (The one we actually want to go back to)
        StringBuilder actualRedirectUri = new StringBuilder(request.getRequestURI());
        String queryString = request.getQueryString();
        if (StringUtils.isNotEmpty(queryString))
        {
            actualRedirectUri.append("?");
            actualRedirectUri.append(queryString);
        }
        
        // saving the actualRedirectUri to enable its use in "OIDCCallbackAction"
        Session session = request.getSession(true);
        session.setAttribute(OAUTH_REDIRECT_URI_SESSION_ATTRIBUTE, actualRedirectUri.toString());
    }
    
    private URI _buildAbsoluteURI(Request request, String path)
    {
        StringBuilder uriBuilder = new StringBuilder()
            .append(request.getScheme())
            .append("://")
            .append(request.getServerName());
        
        if (request.isSecure())
        {
            if (request.getServerPort() != 443)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }
        else
        {
            if (request.getServerPort() != 80)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }

        uriBuilder.append(request.getContextPath());
        uriBuilder.append(path);
        
        return URI.create(uriBuilder.toString());
    }

    public AccessToken requestAccessToken(AuthorizationGrant authorizationGrant) throws IOException
    {
        TokenRequest tokenRequest = new TokenRequest(getTokenEndpointURI(), getClientAuthentication(), authorizationGrant, getScope());
        
        ZonedDateTime requestDate = ZonedDateTime.now();
        HTTPResponse httpResponse = tokenRequest.toHTTPRequest().send();
        return _getAccessTokenFromAuthorizationServerResponse(httpResponse, requestDate);
    }
    
    /**
     * Parse the token response to get the token and store it before returning the newly acquired token
     * @param httpResponse the response
     * @param requestDate the date of the request
     * @return the new token
     * @throws AccessDeniedException if the response doesn't indicate success
     */
    protected AccessToken _getAccessTokenFromAuthorizationServerResponse(HTTPResponse httpResponse, ZonedDateTime requestDate)
    {
        TokenResponse tokenResponse;
        try
        {
            tokenResponse = TokenResponse.parse(httpResponse);
        }
        catch (ParseException e)
        {
            getLogger().error("Token response is invalid. Access will be denied", e);
            throw new AccessDeniedException("Oauth authorization request failed with invalid response. The response was not parseable");
        }
        if (tokenResponse.indicatesSuccess())
        {
            AccessTokenResponse successResponse = tokenResponse.toSuccessResponse();
            
            return storeTokens(successResponse, requestDate);
        }
        else
        {
            ErrorObject errorObject = tokenResponse.toErrorResponse().getErrorObject();
            throw new AccessDeniedException("Oauth authorization request failed with http status '" + errorObject.getHTTPStatusCode()
            + "', code '" + errorObject.getCode()
            + "' and description '" + errorObject.getDescription() + "'.");
        }
    }
    
    /**
     * Store the tokens information for later uses.
     * @param response the tokens returned by the successful token request
     * @param requestDate the request date to compute the expiration date
     * @return the access token
     */
    protected AccessToken storeTokens(AccessTokenResponse response, ZonedDateTime requestDate)
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession();
        
        Tokens tokens = response.getTokens();
        
        
        AccessToken accessToken = tokens.getAccessToken();
        session.setAttribute(OAUTH_ACCESS_TOKEN_SESSION_ATTRIBUTE + "$" + _id, accessToken.toJSONObject().toJSONString());
        session.setAttribute(OAUTH_REFRESH_TOKEN_SESSION_ATTRIBUTE + "$" + _id, tokens.getRefreshToken().toJSONObject().toJSONString());
        if (accessToken.getLifetime() != 0)
        {
            session.setAttribute(OAUTH_ACCESS_TOKEN_EXPIRATION_DATE_SESSION_ATTRIBUTE + "$" + _id, DateUtils.zonedDateTimeToString(requestDate.plusSeconds(accessToken.getLifetime())));
        }
        
        // Store custom parameters if any
        
        Map<String, Object> customParameters = response.getCustomParameters();
        for (String paramName : getCustomParametersName())
        {
            Object param = customParameters.get(paramName);
            // Session attribute must be stored as string for possible serialization
            session.setAttribute(OAUTH_CUSTOM_PARAMETER + "$" + _id + "#" + paramName, JSONValue.toJSONString(param));
        }
        
        return accessToken;
    }
}
