/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.oauth;

import java.io.IOException;
import java.net.URI;
import java.util.Optional;
import java.util.Set;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Redirector;

import org.ametys.runtime.authentication.AccessDeniedException;

import com.nimbusds.oauth2.sdk.AuthorizationGrant;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;

/**
 * Represent an OAuth authorization server
 */
public interface OAuthProvider
{
    /**
     * Get the extension id
     * @return the id
     */
    public String getId();
    
    /**
     * Get the client id provided by the authorization server.
     * @return the client id
     */
    public ClientID getClientID();
    
    /**
     * Get the client authentication to use for new token request
     * @return the authentication
     */
    public ClientAuthentication getClientAuthentication();
    
    /**
     * Get the authorization endpoint for the provider
     * @return the uri
     */
    public URI getAuthorizationEndpointURI();
    
    /**
     * Get the token endpoint for the provider
     * @return the uri
     */
    public URI getTokenEndpointURI();

    /**
     * Get the list of custom parameters returned with the access token.
     * The listed parameters are stored with the access token for later
     * use.
     * @return the list of parameter names
     */
    public Set<String> getCustomParametersName();
    
    /**
     * Get the scope to request to the provider.
     * {@code null} if no scope should be use.
     * @return the scope or {@code null}
     */
    public Scope getScope();
    
    /**
     * Get the stored access token if it exist.
     * 
     * If a token exists, but is expired, this method will silently try to refresh it.
     * @return the access token or empty
     */
    public Optional<AccessToken> getStoredAccessToken();
    
    /**
     * Try to get the currently stored access token for the given provider.
     * If no valid token is available, the method should return {@link Optional#empty()}
     * and try to redirect to the authorization protocol.
     * 
     * @see #getStoredAccessToken() if you don't want to redirect
     * 
     * @param redirector the redirector to use if no token are available.
     * @return the token or {@link Optional#empty()}
     * @throws ProcessingException if an error occurred while redirecting
     * @throws IOException if an error occurred while redirecting
     */
    public Optional<AccessToken> getAccessToken(Redirector redirector) throws ProcessingException, IOException;
    
    /**
     * Indicate that the provider is expecting an authorization response
     * for the provided state.
     * 
     * @param state the state
     * @return true if the provider has initiated a authorization process with this state and the process is not complete yet.
     */
    public boolean isKnownState(State state);

    /**
     * Retrieve the stored value for a custom parameter
     * @param <T> the type of the value to retrieve. This must be a valid JSON type
     * @param parameter the name of the parameter
     * @return the value of the parameter if it exists
     */
    public <T> Optional<T> getStoredCustomParameter(String parameter);

    /**
     * Request a token to an authorization server using an authorization
     * code previously provided by the authorization server.
     * 
     * This method will store the result of the token request in session.
     * See {@link #getStoredAccessToken()} and {@link #getAccessToken(Redirector)}
     *  for how to use the requested token.
     * 
     * @param authorizationGrant the grant to use to request the token
     * @return the new access token
     * @throws IOException if an error occurred while communicating with the token endpoint
     * @throws AccessDeniedException if the response contains error
     * 
     * @see #getStoredAccessToken()
     * @see #getAccessToken(Redirector)
     */
    public AccessToken requestAccessToken(AuthorizationGrant authorizationGrant) throws IOException;
}
