/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.extrausermgt.authentication.oauth;

import java.net.URI;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.AbstractAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.plugins.extrausermgt.oauth.DefaultOauthProvider;
import org.ametys.plugins.extrausermgt.oauth.OAuthProvider;
import org.ametys.plugins.extrausermgt.oauth.OauthProviderExtensionPoint;
import org.ametys.runtime.authentication.AccessDeniedException;

import com.nimbusds.oauth2.sdk.AuthorizationCodeGrant;
import com.nimbusds.oauth2.sdk.AuthorizationResponse;
import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.id.State;

/**
 * Action intended to manage result of an authorize request.
 * 
 * This action will check the response, retrieve the authorization code
 * use it to request an access token and finally store it in session before
 * redirecting to the original request that redirected to the authorize request
 */
public class OAuthCallbackAction extends AbstractAction implements ThreadSafe, Serviceable
{
    private OauthProviderExtensionPoint _oauthEP;

    public void service(ServiceManager manager) throws ServiceException
    {
        _oauthEP = (OauthProviderExtensionPoint) manager.lookup(OauthProviderExtensionPoint.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, List<String>> params = new HashMap<>();
        for (Iterator paramIterator = request.getParameterNames().asIterator(); paramIterator.hasNext();)
        {
            String paramName = (String) paramIterator.next();
            params.put(paramName, Arrays.asList(request.getParameterValues(paramName)));
        }
        
        AuthorizationResponse response = AuthorizationResponse.parse(URI.create(request.getRequestURI()), params);
        
        Session session = request.getSession();
        if (!response.indicatesSuccess())
        {
            ErrorObject errorObject = response.toErrorResponse().getErrorObject();
            throw new AccessDeniedException("Oauth authorization request failed with http status '" + errorObject.getHTTPStatusCode()
            + "', code '" + errorObject.getCode()
            + "' and description '" + errorObject.getDescription() + "'.");
        }
        
        // check that response match the state saved for this request
        checkResponseIntegrity(response, session);
        
        // use the state to retrieve the provider linked to this response
        OAuthProvider provider = _oauthEP.getProviderForState(response.getState());
        
        // get the access token and store it
        AuthorizationCodeGrant grant = new AuthorizationCodeGrant(response.toSuccessResponse().getAuthorizationCode(), URI.create(request.getRequestURI()));
        provider.requestAccessToken(grant);
        
        // we are done with the authorization process.
        // we can now redirect to the original request requiring a protected resource
        String originalRequest = (String) session.getAttribute(DefaultOauthProvider.OAUTH_REDIRECT_URI_SESSION_ATTRIBUTE);
        if (!redirector.hasRedirected())
        {
            redirector.redirect(false, originalRequest);
        }
        
        return EMPTY_MAP;
    }

    /**
     * Check the response integrity
     * @param response the authorize request response
     * @param session the current session
     */
    protected void checkResponseIntegrity(AuthorizationResponse response, Session session)
    {
        State state = (State) session.getAttribute(DefaultOauthProvider.OAUTH_STATE_SESSION_ATTRIBUTE);
        if (state == null || !state.equals(response.getState()))
        {
            throw new AccessDeniedException("Failed to retrieve the authorization code. Oauth state mismatch.");
        }
        else
        {
            // the session state was successfully used to unsure that the response is linked to the current session.
            // we can now remove it as it fulfilled its purpose
            session.removeAttribute(DefaultOauthProvider.OAUTH_STATE_SESSION_ATTRIBUTE);
        }
    }
}
