/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button that is able to check for conditions on extraction rights.
 * @private
 */
Ext.define('Ametys.plugins.extraction.rights.ExtractionController', {
    extend: 'Ametys.ribbon.element.ui.ButtonController',
    
    /**
     * @cfg {String} [enable-on-readaccess-only="false"] If 'true' the button will be disabled as soon as the current user has no read access on extraction selection
     */
    /**
     * @cfg {String} description-no-readaccess The description when the current user has no read access on extraction selection but #cfg-enable-on-readaccess-only is true.
     */
    /**
     * @cfg {String} [enable-on-writeaccess-only="false"] If 'true' the button will be disabled as soon as the current user has no write access on extraction selection
     */
    /**
     * @cfg {String} description-no-writeaccess The description when the current user has no write access on extraction selection but #cfg-enable-on-writeaccess-only is true.
     */
    
    constructor: function (config)
    {
        this.callParent(arguments);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onExtractionModified, this);
    },
    
    additionalErrorDescriptionOnSelectionChanged: function(targets)
    {
        var enabledOnReadAccessOnly = this.getInitialConfig("enable-on-readaccess-only") == 'true';
        if (enabledOnReadAccessOnly && !targets[0].getParameters().canRead)
        {
            return this.getInitialConfig('description-no-readaccess') || '';
        }
        
        var enabledOnWriteAccessOnly = this.getInitialConfig("enable-on-writeaccess-only") == 'true';
        if (enabledOnWriteAccessOnly && !targets[0].getParameters().canWrite)
        {
            return this.getInitialConfig('description-no-writeaccess') || '';
        }

        var enabledOnEditAccessOnly = this.getInitialConfig("enable-on-removeaccess-only") == 'true';
        if (enabledOnEditAccessOnly && !targets[0].getParameters().canDelete)
        {
            return this.getInitialConfig('description-no-removeaccess') || '';
        }

        var enabledOnRenameAccessOnly = this.getInitialConfig("enable-on-renameaccess-only") == 'true';
        if (enabledOnRenameAccessOnly && !targets[0].getParameters().canRename)
        {
            return this.getInitialConfig('description-no-renameaccess') || '';
        }

        var enabledOnEditWorkflowAvailableOnly = this.getInitialConfig("enable-on-edit-workflow-available-only") == 'true';
        if (enabledOnEditWorkflowAvailableOnly)
        {
            var contentTargets = targets[0].getSubtargets("content");
            var editWorkflowActionId = parseInt(this.getInitialConfig("editWorkflowActionId"))
            if(contentTargets.length > 0 && contentTargets[0].getParameters().availableActions.indexOf(editWorkflowActionId) == -1)
            {
                return this.getInitialConfig('description-workflow-unavailable') || '';
            }
        }
    },
    
    /**
     * Listener on modified message.
     * Update the state of the controller accordingly.
     * @param {Ametys.message.Message} message the message of type modified.
     * @private
     */
    _onExtractionModified: function(message)
    {
        this.updateTargetsInCurrentSelectionTargets (message);
    },
    
    areSameTargets: function (target1, target2)
    {
        if (target1.getParameters().path && target2.getParameters().path)
        {
            return target1.getParameters().path == target2.getParameters().path;
        }
        return false;
    }
});
