/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.LifecycleHelper;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.component.CountExtractionComponent;
import org.ametys.plugins.extraction.component.ExtractionComponent;
import org.ametys.plugins.extraction.component.MappingQueryExtractionComponent;
import org.ametys.plugins.extraction.component.QueryExtractionComponent;
import org.ametys.plugins.extraction.component.ThesaurusExtractionComponent;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariableType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * This class reads the extraction definition file
 */
public class ExtractionDefinitionReader extends AbstractLogEnabled implements Component, Contextualizable, Serviceable
{
    /** The component role. */
    public static final String ROLE = ExtractionDefinitionReader.class.getName();
    
    private Context _context;
    private ServiceManager _serviceManager;
    
    private UserHelper _userHelper;
    private AmetysObjectResolver _ametysResolver;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _serviceManager = manager;
        
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _ametysResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    /**
     * Read the extraction definition file
     * @param file extraction definition file
     * @return the extraction components parsed in the definition file
     * @throws Exception if an error occurs
     */
    public Extraction readExtractionDefinitionFile(File file) throws Exception
    {
        long startTime = -1;
        if (getLogger().isDebugEnabled())
        {
            startTime = System.currentTimeMillis();
            getLogger().debug("Reading definition file");
        }
        
        assert file != null;
        Configuration configuration = new DefaultConfigurationBuilder().buildFromFile(file);
        
        Extraction extraction = new Extraction(file.getName());
        
        _readExtractionDescription(configuration, extraction);
        _readExtractionAuthor(configuration, extraction);
        _readVariablesDefinition(configuration, extraction);
        _readExtractionDefinitionFile(configuration, extraction);
        
        if (getLogger().isDebugEnabled())
        {
            long endTime = System.currentTimeMillis();
            getLogger().debug("Read definition file in " + (endTime - startTime) + "ms");
        }
        
        return extraction;
    }
    
    /**
     * Read the extraction definition file
     * @param file extraction definition file
     * @return the extraction components parsed in the definition file
     * @throws Exception if an error occurs
     */
    public Extraction readVariablesDefinitionsInExtractionDefinitionFile(File file) throws Exception
    {
        assert file != null;
        
        Configuration configuration = new DefaultConfigurationBuilder().buildFromFile(file);
        Extraction extraction = new Extraction(file.getName());
        _readVariablesDefinition(configuration, extraction);
        
        return extraction;
    }
    
    private void _readExtractionDescription(Configuration configuration, Extraction extraction)
    {
        String descriptionId = configuration.getChild(ExtractionConstants.DESCRIPTION_TAG, true)
                .getAttribute(ExtractionConstants.DESCRIPTION_IDENTIFIER_ATTRIBUTE_NAME, null);
        
        if (StringUtils.isNotBlank(descriptionId))
        {
            try
            {
                _ametysResolver.resolveById(descriptionId);
                extraction.setDescriptionId(descriptionId);
            }
            catch (AmetysRepositoryException e)
            {
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn("Invalid extraction description " + descriptionId + " for configuration " + configuration.getLocation(), e);
                }
            }
        }
    }
    
    private void _readExtractionAuthor(Configuration configuration, Extraction extraction) throws ConfigurationException
    {
        Configuration author = configuration.getChild(ExtractionConstants.AUTHOR_TAG, false);
        if (author != null)
        {
            extraction.setAuthor(_userHelper.xml2userIdentity(author));
        }
    }
    
    private void _readVariablesDefinition(Configuration configuration, Extraction extraction) throws ConfigurationException
    {
        for (Configuration child : configuration.getChildren())
        {
            switch (child.getName())
            {
                case ExtractionConstants.OPTIONAL_COLUMNS_TAG:
                    extraction.setDisplayOptionalColumnsNames(getDisplayOptionalColumnNames(child));
                    break;
                case ExtractionConstants.CLAUSES_VARIABLES_TAG:
                    extraction.setClausesVariables(getClausesVariables(child));
                    break;
                default:
                    // Do nothing, we only check variables definitions
            }
        }
    }
    
    private void _readExtractionDefinitionFile(Configuration configuration, Extraction extraction) throws Exception
    {
        for (Configuration child : configuration.getChildren())
        {
            switch (child.getName())
            {
                case ExtractionConstants.QUERY_COMPONENT_TAG:
                case ExtractionConstants.THESAURUS_COMPONENT_TAG:
                case ExtractionConstants.COUNT_COMPONENT_TAG:
                case ExtractionConstants.MAPPING_QUERY_COMPONENT_TAG:
                    ExtractionComponent component = _processExtractionComponent(child);
                    extraction.addExtractionComponent(component);
                    break;
                default:
                    // Do nothing
            }
        }
    }

    private List<String> getDisplayOptionalColumnNames(Configuration configuration) throws ConfigurationException
    {
        List<String> names = new ArrayList<>();
        for (Configuration nameConfiguration : configuration.getChildren("name"))
        {
            names.add(nameConfiguration.getValue());
        }
        return names;
    }
    
    private List<ClausesVariable> getClausesVariables(Configuration configuration) throws ConfigurationException
    {
        List<ClausesVariable> variables = new ArrayList<>();
        
        for (Configuration variableConfiguration : configuration.getChildren("variable"))
        {
            // Name
            String name = variableConfiguration.getAttribute("name", null);
            if (null == name)
            {
                throw new ConfigurationException("A clauses variable is not well defined, name is mandatory.");
            }
            
            // Type
            String typeAsString = variableConfiguration.getAttribute("type", null);
            // For legacy purpose, if no type is defined, type is the SELECT_CONTENTS one
            ClausesVariableType type = typeAsString != null ? ClausesVariableType.fromStringValue(typeAsString) : ClausesVariableType.SELECT_CONTENTS;
            
            // Content type identifiers
            List<String> contentTypeIds = _getClausesVariableContentTypeIds(variableConfiguration, type);
            
            // Search model identifier
            Optional<String> searchModelId = _getClausesVariableSearchModelId(variableConfiguration);
            
            // Solr request
            Optional<String> solrRequest = _getClausesVariableSolrRequest(variableConfiguration);
            
            variables.add(new ClausesVariable(name, type, contentTypeIds, searchModelId, solrRequest));
        }
        
        return variables;
    }
    
    private List<String> _getClausesVariableContentTypeIds(Configuration variableConfiguration, ClausesVariableType clausesVariableType) throws ConfigurationException
    {
        List<String> contentTypeIds = new ArrayList<>();
        
        // Look for legacy content type
        Optional.ofNullable(variableConfiguration.getAttribute("contentType", null))
                .ifPresent(contentTypeId -> contentTypeIds.add(contentTypeId));
        
        // If there is no legacy content type, get the content types' list
        if (contentTypeIds.isEmpty())
        {
            Configuration contentTypesConf = variableConfiguration.getChild("content-types");
            for (Configuration contentTypeConf : contentTypesConf.getChildren("content-type"))
            {
                String id = contentTypeConf.getAttribute("id");
                contentTypeIds.add(id);
            }
        }
        
        // Check that there is max one content type for SELECT_CONTENTS variable
        if (ClausesVariableType.SELECT_CONTENTS.equals(clausesVariableType) && contentTypeIds.size() > 1)
        {
            throw new ConfigurationException("Only one content type id can be provided for variables of type selectContents", variableConfiguration);
        }
        
        // Check that all referenced content types are existing 
        for (String contentTypeId : contentTypeIds)
        {
            if (!_contentTypeExtensionPoint.hasExtension(contentTypeId))
            {
                throw new ConfigurationException("Only one content type id can be provided for variables of type selectContents", variableConfiguration);
            }
        }
        
        return contentTypeIds;
    }
    
    private Optional<String> _getClausesVariableSearchModelId(Configuration variableConfiguration)
    {
        String searchModelId = variableConfiguration.getChild("search-model-id").getValue(null);

        return Optional.ofNullable(searchModelId)
                       .filter(StringUtils::isNotEmpty);
    }
    
    private Optional<String> _getClausesVariableSolrRequest(Configuration variableConfiguration)
    {
        String solrRequest = variableConfiguration.getChild("solr-request").getValue(null);

        return Optional.ofNullable(solrRequest)
                       .filter(StringUtils::isNotEmpty);
    }

    private ExtractionComponent _processExtractionComponent(Configuration componentConfiguration) throws Exception
    {
        ExtractionComponent component = null;
        switch (componentConfiguration.getName())
        {
            case "query":
                component = new QueryExtractionComponent();
                break;
            case "count":
                component = new CountExtractionComponent();
                break;
            case "thesaurus":
                component = new ThesaurusExtractionComponent();
                break;
            case "mapping-query":
                component = new MappingQueryExtractionComponent();
                break;
            default:
                // do nothing
                break;
        }

        if (component != null)
        {
            LifecycleHelper.setupComponent(component, getLogger(), _context, _serviceManager, componentConfiguration);
            for (Configuration child : componentConfiguration.getChildren())
            {
                ExtractionComponent subComponent = _processExtractionComponent(child);
                if (null != subComponent)
                {
                    component.addSubComponent(subComponent);
                }
            }
        }
        return component;
    }
}
