/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline.impl;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.stream.Collector;

import javax.xml.transform.Templates;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamSource;

import org.apache.cocoon.components.xslt.TraxErrorListener;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.ContentHandler;

import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.pipeline.Pipeline;
import org.ametys.plugins.extraction.execution.pipeline.PipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.PipelineSerializerModel;
import org.ametys.plugins.extraction.execution.pipeline.Pipelines;

class PipelineImpl implements Pipeline
{
    private static final Logger __LOGGER = LoggerFactory.getLogger(Pipeline.class);
    private static final org.apache.avalon.framework.logger.Logger __AVALON_LOGGER = new SLF4JLoggerAdapter(__LOGGER);
    
    private PipelineDescriptor _desc;
    private OutputStream _out;
    private SourceResolver _resolver;
    private List<Source> _sources = new ArrayList<>();
    private TransformerHandler _firstHandler;
    private TransformerHandler _lastHandler;
    private PipelineSerializer _serializer;

    PipelineImpl(PipelineDescriptor desc, OutputStream out, SourceResolver resolver)
    {
        _desc = desc;
        _out = out;
        _resolver = resolver;
    }
    
    @Override
    public ContentHandler getHandler() throws Exception
    {
        if (_firstHandler != null || _lastHandler != null)
        {
            throw new IllegalStateException("Pipeline was already opened.");
        }
        
        SAXTransformerFactory stf = Pipelines.getSaxTransformerFactory();
        if (_desc.getStylesheets().isEmpty())
        {
            _firstHandler = stf.newTransformerHandler();
            _lastHandler = _firstHandler;
        }
        else
        {
            _getTransformerHandler(stf);
        }
        
        PipelineSerializerModel serializerModel = _desc.getSerializerModel();
        _serializer = serializerModel.newSerializer(_lastHandler, _out, _desc.getOutputParameters());
        
        _serializer.prepare();
        return _firstHandler;
    }
    
    // case _xslt is not empty
    private void _getTransformerHandler(SAXTransformerFactory stf)
    {
        final String suffix = ExtractionConstants.XSLT_DIR + "/";
        Queue<TransformerHandler> xslts = _desc.getStylesheets().stream()
                .map(suffix::concat)
                .map(LambdaUtils.wrap(_resolver::resolveURI))
                .map(LambdaUtils.wrap(source -> _newTransformerHandler(stf, source)))
                .collect(Collector.of(
                    ArrayDeque::new, 
                    (deque, th) -> deque.add(th), 
                    (deque1, deque2) ->
                    {
                        throw new IllegalStateException("Should not be parallel");
                    }
                ));
        
        _firstHandler = xslts.element();
        
        // chain
        while (true)
        {
            TransformerHandler currentHandler = xslts.remove();
            if (xslts.isEmpty())
            {
                // it is last => set result
                _lastHandler = currentHandler;
                break;
            }
            else
            {
                // Set next one as the result of current
                currentHandler.setResult(new SAXResult(xslts.element()));
            }
        }
    }
    
    private TransformerHandler _newTransformerHandler(SAXTransformerFactory stf, Source source) throws TransformerConfigurationException
    {
        // Need to keep sources references
        _sources.add(source);
        String uri = source.getURI();
        _checkExists(source, uri);
        FileSource fileSource = _checkFileSource(source, uri);
        
        // Create TransformerHandler with custom error listener
        StreamSource streamSource = new StreamSource(fileSource.getFile());
        Templates templates = stf.newTemplates(streamSource);
        TransformerHandler th = stf.newTransformerHandler(templates);
        TraxErrorListener errorListener = new TraxErrorListener(__AVALON_LOGGER, uri);
        th.getTransformer().setErrorListener(errorListener);
        
        return th;
    }
    
    private void _checkExists(Source s, String uri)
    {
        boolean exists = s.exists();
        if (!exists)
        {
            throw new IllegalArgumentException("XSL file '" + uri + " does not exist. The pipeline cannot be executed.");
        }
    }
    
    private FileSource _checkFileSource(Source source, String uri)
    {
        if (source instanceof FileSource)
        {
            return (FileSource) source;
        }
        throw new IllegalArgumentException("Source '" + uri + " is not a file source. The pipeline cannot be executed.");
    }
    
    @Override
    public void serialize() throws Exception
    {
        _serializer.serialize();
    }

    @Override
    public void close() throws IOException
    {
        _release();
        if (_serializer != null)
        {
            _serializer.close();
        }
    }
    
    private void _release()
    {
        if (_resolver != null)
        {
            _sources.forEach(_resolver::release);
        }
    }
}
