/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.GroupSearchContentHelper;
import org.ametys.cms.search.cocoon.GroupSearchContent;
import org.ametys.core.util.StringUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;
import org.ametys.runtime.model.ViewItemContainer;

/**
 * This class represents an extraction component with grouping fields
 */
public abstract class AbstractGroupExtractionComponent extends AbstractSolrExtractionComponent
{
    
    /** The list of grouping fields */
    protected List<String> _groupingFields = new ArrayList<>();

    private GroupSearchContentHelper _groupSearchContentHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _groupSearchContentHelper = (GroupSearchContentHelper) serviceManager.lookup(GroupSearchContentHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        String groupingFieldsAsString = configuration.getChild("grouping-fields").getValue("");
        _groupingFields.addAll(StringUtils.stringToCollection(groupingFieldsAsString));
    }
    
    @Override
    public void prepareComponentExecution(ExtractionExecutionContext context) throws Exception
    {
        super.prepareComponentExecution(context);
        
        for (String groupingField : _groupingFields)
        {
            // get attribute type just to throw an Exception if the field is not available for content types 
            this._getAttributeTypeId(groupingField, _contentTypes);
        }
    }

    /**
     * Organizes contents in groups
     * @param contents The contents to organize
     * @param defaultLocale The default locale for localized values. Only useful if grouping by multilingual contents or multilingual string
     * @return groups of content
     */
    protected GroupSearchContent organizeContentsInGroups(Iterable<Content> contents, Locale defaultLocale)
    {
        Set<ContentType> baseContentTypes = _contentTypesHelper.getCommonAncestors(_contentTypes).stream()
                .map(_contentTypeExtensionPoint::getExtension)
                .collect(Collectors.toSet());
        return _groupSearchContentHelper.organizeContentsInGroups(contents, _groupingFields, baseContentTypes, defaultLocale);
    }
    
    /**
     * Recursive method that saxes group of contents
     * First level (level=0) will not create a xml group (usually, 1st group is just here to handle a list)
     * @param contentHandler result document
     * @param group group to sax
     * @param level current level (start with zero)
     * @param context execution context
     * @param resultItems result items to sax
     * @throws Exception if a error occurred
     */
    protected void saxGroup(ContentHandler contentHandler, GroupSearchContent group, int level, ExtractionExecutionContext context, ViewItemContainer resultItems) throws Exception
    {
        if (level > 0)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("level", String.valueOf(level));
            attrs.addCDATAAttribute("fullModelPath", group.getGroupFieldPath());
            attrs.addCDATAAttribute("value", group.getGroupName());
            XMLUtils.startElement(contentHandler, "group", attrs);
        }
        
        for (GroupSearchContent groupSearchContent : group.getSubList())
        {
            saxGroup(contentHandler, groupSearchContent, level + 1, context, resultItems);
        }

        if (!group.getContents().isEmpty() || group.getSubList().isEmpty())
        {
            // Generate SAX events for contents if there are contents or if there is no sub groups
            saxContents(contentHandler, context, resultItems, group.getContents());
        }
        
        if (level > 0)
        {
            XMLUtils.endElement(contentHandler, "group");
        }
    }
    
    /**
     * Sax a content
     * @param contentHandler result document
     * @param context execution context
     * @param resultItems result fields to sax
     * @param contents contents to sax
     * @throws Exception if an error occurs
     */
    protected abstract void saxContents(ContentHandler contentHandler, ExtractionExecutionContext context, ViewItemContainer resultItems, List<Content> contents) throws Exception;
    
    @Override
    public Map<String, Object> getComponentDetailsForTree()
    {
        Map<String, Object> details = super.getComponentDetailsForTree();
        
        @SuppressWarnings("unchecked")
        Map<String, Object> data = (Map<String, Object>) details.get("data");
        data.put("groupingFields", String.join(ExtractionConstants.STRING_COLLECTIONS_INPUT_DELIMITER, this.getGroupingFields()));
        
        return details;
    }

    /**
     * Retrieves the component grouping fields
     * @return The component grouping fields
     */
    public List<String> getGroupingFields()
    {
        return _groupingFields;
    }

    /**
     * Add grouping fields to the component
     * @param groupingFields Array of the grouping fields to add
     */
    public void addGroupingFields(String... groupingFields)
    {
        _groupingFields.addAll(Arrays.asList(groupingFields));
    }
}
