/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.extraction.component.ExtractionComponent;

/**
 * Object representing the extraction definition file content
 */
public class Extraction
{    
    /**
     * Rights profiles
     */
    public enum ExtractionProfile
    {
        /** Read access */
        READ_ACCESS,
        /** Write access */
        WRITE_ACCESS,
        /** Right access */
        RIGHT_ACCESS;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
        
    }
    
    /**
     * Types of clauses variables
     */
    public enum ClausesVariableType
    {
        /** Solr request type: the user will give a solr request to select the contents */
        SOLR_REQUEST("solrRequest"),
        /** Select contents type: the user will select contents in dropdown list */ 
        SELECT_CONTENTS("selectContents");
        
        private final String _stringValue;
        
        ClausesVariableType(String stringValue)
        {
            _stringValue = stringValue;
        }
        
        /**
         * Retrieves the enum's value
         * @return the enum's value
         */
        public String getStringValue()
        {
            return _stringValue;
        }
        
        /**
         * Retrieves the {@link ClausesVariableType} corresponding to the given string value
         * @param stringValue the string value
         * @return the {@link ClausesVariableType} corresponding to the given string value
         * @throws IllegalArgumentException if the given string value is not an available type
         */
        public static ClausesVariableType fromStringValue(String stringValue) throws IllegalArgumentException
        {
            for (ClausesVariableType type : ClausesVariableType.values())
            {
                if (type.getStringValue().equals(stringValue))
                {
                    return type;
                }
            }
            
            throw new IllegalArgumentException("'" + stringValue + "' is not an available type of clauses variable");
        }
    }
    
    private String _fileName;
    private String _descriptionId;
    private List<ExtractionComponent> _extractionComponents;
    private List<String> _displayOptionalColumnsNames;
    private List<ClausesVariable> _clausesVariables;
    private UserIdentity _author;
    
    /**
     * Constructor
     * @param fileName The file name
     */
    public Extraction(String fileName)
    {
        _fileName = fileName;
    }
    
    /**
     * Stores a clauses variable data
     * @param name the name of the variable
     * @param type the type of the variable (SOLR_REQUEST or SELECT_CONTENTS)
     * @param contentTypeIds the content type identifiers to restrict the contents to select. Only one content type should be provided for SELECT_CONTENT type of variables
     * @param searchModelId the search model identifier to use for select contents
     * @param solrRequest the solr request use to limit contents to select
     */
    public record ClausesVariable(String name, ClausesVariableType type, List<String> contentTypeIds, Optional<String> searchModelId, Optional<String> solrRequest) { /* empty */ }
    
    /**
     * Get the file name of this extraction
     * @return the file name
     */
    public String getFileName()
    {
        return _fileName;
    }
    
    /**
     * Retrieves the identifier of the extraction's description
     * @return the identifier of the extraction's description
     */
    public String getDescriptionId()
    {
        return _descriptionId;
    }
    
    /**
     * Set the identifier of the extraction's description
     * @param descriptionId the identifier to set
     */
    public void setDescriptionId(String descriptionId)
    {
        _descriptionId = descriptionId;
    }
    
    /**
     * Retrieves the list of the extraction components
     * @return The extraction components 
     */
    public List<ExtractionComponent> getExtractionComponents()
    {
        return null != _extractionComponents ? _extractionComponents : new ArrayList<>();
    }
    
    /**
     * Add an extraction component 
     * @param extractionComponent The extraction component to add
     */
    public void addExtractionComponent(ExtractionComponent extractionComponent)
    {
        if (null == _extractionComponents)
        {
            _extractionComponents = new ArrayList<>();
        }
        _extractionComponents.add(extractionComponent);
    }
    
    /**
     * Retrieves the list of variables names controlling display of optional columns
     * @return The variables names
     */
    public List<String> getDisplayOptionalColumnsNames()
    {
        return null != _displayOptionalColumnsNames ? _displayOptionalColumnsNames : new ArrayList<>();
    }
    
    /**
     * Set variables names controlling display of optional columns
     * @param displayOptionalColumnsNames the variables for the optional columns to set
     */
    public void setDisplayOptionalColumnsNames(List<String> displayOptionalColumnsNames)
    {
        _displayOptionalColumnsNames = displayOptionalColumnsNames;
    }
    
    /**
     * Retrieves the list of variables to use in clauses
     * @return A List containing clauses variables
     */
    public List<ClausesVariable> getClausesVariables()
    {
        return null != _clausesVariables ? _clausesVariables : new ArrayList<>();
    }
    
    /**
     * Set variables to use in clauses
     * @param clausesVariables A Map containing variables to set
     */
    public void setClausesVariables(List<ClausesVariable> clausesVariables)
    {
        _clausesVariables = clausesVariables;
    }

    /**
     * Get the author of the extraction
     * @return The author
     */
    public UserIdentity getAuthor ()
    {
        return _author;
    }
    
    /**
     * Set the author of this extraction.
     * @param author the author
     */
    public void setAuthor(UserIdentity author)
    {
        _author = author;
    }
}
