/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Function;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.container.ContainerUtil;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.xml.sax.SAXException;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.pipeline.impl.ConfigurablePipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.impl.NoOpPipelineDescriptor;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Manager of {@link PipelineDescriptor}s
 */
public class PipelineManager extends AbstractLogEnabled implements Component, Serviceable, Initializable, Disposable
{
    /** The Avalon role. */
    public static final String ROLE = PipelineManager.class.getName();
    
    // '*' char is appended as it is a forbidden character for file names, it ensures the uniqueness of the id
    private static final String __NO_OP_PIPELINE_ID = NoOpPipelineDescriptor.class.getName() + "*";
    
    private static final String __PIPELINE_DESCRIPTOR_CACHE = "plugin-extraction.PipelineDescriptor";
    
    private PipelineDescriptor _noOpPipelineDesc;
    private Map<String, Long> _lastReading = new HashMap<>();

    private AbstractCacheManager _cacheManager;
    private SourceResolver _resolver;
    private PipelineSerializerModelExtensionPoint _serializerEP;

    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _resolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _serializerEP = (PipelineSerializerModelExtensionPoint) manager.lookup(PipelineSerializerModelExtensionPoint.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        PipelineSerializerModel xmlSerializer = _serializerEP.getExtension("xml");
        _noOpPipelineDesc = new NoOpPipelineDescriptor(xmlSerializer);
        
        _cacheManager.createMemoryCache(__PIPELINE_DESCRIPTOR_CACHE,
                new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_CACHE_PIPELINE_DESCRIPTOR_LABEL"),
                new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_CACHE_PIPELINE_DESCRIPTOR_DESCRIPTION"),
                true,
                null);
    }
    
    @Override
    public void dispose()
    {
        _getPipelineDescriptorCache().invalidateAll();
        _lastReading.clear();
    }
    
    /**
     * Returns <code>true</code> if the {@link PipelineDescriptor} for asked path exists
     * @param path The path of the {@link PipelineDescriptor}
     * @return <code>true</code> if the asked pipeline exists
     * @throws IOException if an I/O exception occurred when accessing the root folder of pipeline files
     */
    public boolean has(String path) throws IOException
    {
        return get(path) != null; 
    }
    
    /**
     * Gets the {@link PipelineDescriptor} for the given path
     * @param path The path of the pipeline
     * @return the request {@link PipelineDescriptor}, or null if not found
     * @throws IOException if an I/O exception occurred when accessing the root folder of pipeline files
     */
    public PipelineDescriptor get(Path path) throws IOException
    {
        return get(path.toString());
    }
    
    /**
     * Gets the {@link PipelineDescriptor} for the given path
     * @param path The path of the pipeline
     * @return the request {@link PipelineDescriptor}, or null if not found
     * @throws IOException if an I/O exception occurred when accessing the root folder of pipeline files
     */
    public PipelineDescriptor get(String path) throws IOException
    {
        if (__NO_OP_PIPELINE_ID.equals(path))
        {
            return _noOpPipelineDesc;
        }
        return _readAndGetPipeline(path);
    }
    
    /**
     * Gets the available pipelines for the given extraction
     * @param extractionId the extraction's definition file name
     * @return the available pipelines
     * @throws IOException if an I/O exception occurred when accessing the root folder of pipeline files
     */
    @Callable (rights = "Extraction_Rights_ExecuteExtraction")
    public Map<String, Object> getAvailablePipelines(String extractionId) throws IOException
    {
        if (extractionId != null)
        {
            String decodedExtractionId = URIUtils.decode(extractionId);
            List<Map<String, Object>> enumeration = _getJsonEnumeration(URIUtils.decode(decodedExtractionId));
            return Map.of("pipelines", enumeration);
        }
           
        return Map.of();
    }
    
    private List<Map<String, Object>> _getJsonEnumeration(String extractionId) throws IOException
    {
        List<Map<String, Object>> enumeration = new ArrayList<>();
        _readAll();
        
        for (Entry<String , PipelineDescriptor> entry : _getPipelineDescriptorCache().asMap().entrySet())
        {
            PipelineDescriptor pipeline = entry.getValue();
            ExtractionMatcher matcher = pipeline.getExtractionMatcher();
            if (matcher.isHandled(extractionId))
            {
                enumeration.add(Map.of(
                        "value", entry.getKey(), 
                        "text", pipeline.getLabel()));
            }
        }
        
        return enumeration;
    }
    
    /**
     * Gets the id of the default {@link PipelineDescriptor}
     * @return the id of the default {@link PipelineDescriptor}
     */
    public String getDefaultPipeline()
    {
        return __NO_OP_PIPELINE_ID;
    }
    
    private synchronized PipelineDescriptor _readAndGetPipeline(String path) throws IOException
    {
        Path pipelinePath = _absolutePath(path);
        if (Files.exists(pipelinePath))
        {
            Pair<String, PipelineDescriptor> readPipeline = _readPipeline(pipelinePath);
            if (readPipeline != null)
            {
                PipelineDescriptor pipeline = readPipeline.getRight();
                String id = readPipeline.getLeft();
                _getPipelineDescriptorCache().put(id, pipeline);
                return pipeline;
            }
        }
        return null;
    }
    
    private synchronized void _readAll() throws IOException
    {
        _readFromFolder(_absolutePath(StringUtils.EMPTY))
            .forEach(pair -> _getPipelineDescriptorCache().put(pair.getKey(), pair.getValue()));
        
        // Add the default pipeline
        _getPipelineDescriptorCache().put(__NO_OP_PIPELINE_ID, _noOpPipelineDesc);
    }
    
    private Path _absolutePath(String relativePath) throws IOException
    {
        FileSource pipelinesDirScr = null;
        FileSource src = null;
        try
        {
            pipelinesDirScr = (FileSource) _resolver.resolveURI(ExtractionConstants.PIPELINES_DIR);
            pipelinesDirScr.getFile().mkdirs();
            
            src = (FileSource) _resolver.resolveURI(ExtractionConstants.PIPELINES_DIR + relativePath);
            File file = src.getFile();
            return file.toPath();
        }
        finally
        {
            _resolver.release(pipelinesDirScr);
            _resolver.release(src);
        }
    }
    
    private Stream<Pair<String, PipelineDescriptor>> _readFromFolder(Path folderPath) throws IOException
    {
        return Files.list(folderPath)
            .map(LambdaUtils.wrap(this::_readFromFile))
            .flatMap(Function.identity());
    }
    
    private Stream<Pair<String, PipelineDescriptor>> _readFromFile(Path path) throws IOException
    {
        return Files.isDirectory(path) ? _readFromFolder(path) : Stream.ofNullable(_readPipeline(path));
    }
    
    private Pair<String, PipelineDescriptor> _readPipeline(Path pipelinePath) throws IOException
    {
        String id = _getPipelineIdFromAbsolutePath(pipelinePath);
        File pipelineFile = pipelinePath.toFile();
        long lastModified = (pipelineFile.lastModified() / 1000) * 1000; // second precision for Linux file systems 
        if (lastModified > _lastReading.getOrDefault(id, -1L))
        {
            // cache is out of date => need to read
            try
            {
                Configuration conf = new DefaultConfigurationBuilder().buildFromFile(pipelineFile);
                PipelineDescriptor pipeline = new ConfigurablePipelineDescriptor(id, _resolver, _serializerEP);
                ContainerUtil.configure(pipeline, conf);
                return Pair.of(id, pipeline);
            }
            catch (ConfigurationException | SAXException | IOException e)
            {
                getLogger().error("File '{}' could not be parsed as an extraction pipeline.", pipelinePath, e);
                return null;
            }
            finally
            {
                _lastReading.put(id, _now());
            }
        }
        else
        {
            // cache is up to date
            PipelineDescriptor pipeline = _getPipelineDescriptorCache().get(id);
            return pipeline == null ? null : Pair.of(id, pipeline);
        }
    }
    
    private String _getPipelineIdFromAbsolutePath(Path pipelineAbsolutePath) throws IOException
    {
        FileSource pipelinesDirScr = null;
        try
        {
            pipelinesDirScr = (FileSource) _resolver.resolveURI(ExtractionConstants.PIPELINES_DIR);
            File pipelinesFile = pipelinesDirScr.getFile();
            Path pipelineRelativePath = pipelinesFile.toPath().relativize(pipelineAbsolutePath);
            return pipelineRelativePath.toString();
        }
        finally
        {
            _resolver.release(pipelinesDirScr);
        }
    }
    
    private static Long _now()
    {
        return Instant.now().truncatedTo(ChronoUnit.SECONDS).toEpochMilli();
    }
    
    private Cache<String, PipelineDescriptor> _getPipelineDescriptorCache()
    {
        return _cacheManager.get(__PIPELINE_DESCRIPTOR_CACHE);
    }
}
