/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

AmetysFlipbook = {
	
	book: null,
	
	selector: null,
	
	init: function(selector, options)
	{
		options = options || {};
		
		AmetysFlipbook.selector = selector;
		
		AmetysFlipbook.book = new St.PageFlip(
	        $j(AmetysFlipbook.selector).get(0),
	        {
	            width: options.width || 600, // base page width
	            height: options.height || 800, // base page height
				
				//usePortrait: options.usePortrait === true, // if true display single page even on large screens (use size = fixed)
				showCover: options.showCover !== false, // default true
	            size: options.size || "stretch",
				autoSize: options.autoSize !== false, // default true
								
	            // set threshold values:
	            minWidth: options.minWidth || 315,
	            maxWidth: options.maxWidth || screen.width * 0.8,
	            minHeight: options.minHeight || 200,
	            maxHeight: options.maxHeight || screen.height * 0.8,
				
				showPageCorners: options.showPageCorners !== false, // default true
	            maxShadowOpacity: options.maxShadowOpacity || 0.5, // Half shadow intensity
	            mobileScrollSupport: false, // disable content scrolling  when touching a book on mobile devices,
				flippingTime: options.flippingTime || 1000, // page flip duration
				disableFlipByClick: true // disable mouse event to support zoom on double-click
	        }
	    );
		
		// Load pages
	    AmetysFlipbook.book.loadFromHTML(document.querySelectorAll(".page"));
		
		// Init page count
	    $j('[data-flipbook-role="page-total"]').html(AmetysFlipbook.book.getPageCount());

		// Handle navigation buttons
		$j('[data-flipbook-role="first"]').on("click", () => {
	        AmetysFlipbook.book.flip(0); // Turn to the first page (with animation)
	    });
		
	    $j('[data-flipbook-role="prev"]').on("click", () => {
	        AmetysFlipbook.book.flipPrev(); // Turn to the previous page (with animation)
	    });
		
	    $j('[data-flipbook-role="next"]').on("click", () => {
	        AmetysFlipbook.book.flipNext(); // Turn to the next page (with animation)
	    });
		
		$j('[data-flipbook-role="last"]').on("click", () => {
	        AmetysFlipbook.book.flip(AmetysFlipbook.book.getPageCount() -1); // Turn to the last page (with animation)
	    });
		
		// Handle prev/next navigation on zoom mode
		$j('[data-flipbook-role="zoom-prev"]').on("click", () => {
			let index = AmetysFlipbook.zoomedPage - 1;
			AmetysFlipbook.zoomIn($j(`[data-flipbook-page="${index}"]`));
	        AmetysFlipbook.book.turnToPage(index-1); // Turn to the previous page (without animation)
	    });
		
		$j('[data-flipbook-role="zoom-next"]').on("click", () => {
			let index = AmetysFlipbook.zoomedPage + 1;
			AmetysFlipbook.zoomIn($j(`[data-flipbook-page="${index}"]`));
	        AmetysFlipbook.book.turnToPage(index-1); // Turn to the previous page (without animation)
	    });
				
		// Zoom-in on page double-click (unavailable on mobile device)
		$j('.page-image').on("dblclick", (e) => {
			AmetysFlipbook.zoomIn(e.target);
		})
		
		$j('[data-flipbook-role="zoom-in"]').on('click', (e) => {
			AmetysFlipbook.zoomIn();
		})
		
		// Zoom-out on zoomed image click
		$j('.zoom-result img').on('click', (e) => {
			// Zoom out
			AmetysFlipbook.zoomOut(e);
		});
		$j('[data-flipbook-role="zoom-out"]').on("click", (e) => {
			AmetysFlipbook.zoomOut(e);
	    });
		
		function _updateNavigation()
		{
			// Update current page(s)
			let orientation = AmetysFlipbook.book.getOrientation();
			let pagesCollection = AmetysFlipbook.book.getPageCollection();
			let spreadIndex = orientation == 'landscape' ? pagesCollection.landscapeSpread : pagesCollection.portraitSpread;
			let currentPages = spreadIndex[pagesCollection.currentSpreadIndex];
			currentPages = $j.map(currentPages, function(v,i) {return v+1;});
			$j('[data-flipbook-role="page-current"]').html(currentPages.join(" - "));
			
			// Update status of pagination buttons
			$j('[data-flipbook-role="first"]').prop('disabled', pagesCollection.currentSpreadIndex == 0);
			$j('[data-flipbook-role="prev"]').prop('disabled', pagesCollection.currentSpreadIndex == 0)
			$j('[data-flipbook-role="next"]').prop('disabled', pagesCollection.currentSpreadIndex == spreadIndex.length -1);
			$j('[data-flipbook-role="last"]').prop('disabled', pagesCollection.currentSpreadIndex == spreadIndex.length -1);
		}
			
		// initialize current page display and pagination buttons status
		_updateNavigation();

	    // triggered by page turning
	    AmetysFlipbook.book.on("flip", (e) => {
			
			// Update current page in thumbnails navigation
			if (!AmetysFlipbook.suspendCurrentPageUpdate)
			{
				let currentPage = e.data + 1;
				$j('[data-flipbook-thumbnail]').removeAttr('aria-current');
				$j(`[data-flipbook-thumbnail="${currentPage}"]`).attr('aria-current', 'true');
			}
			
			_updateNavigation();
	    });
		
		$j("body").on('keydown', AmetysFlipbook.closeThumbnailNavigationonEsc);
	},
	
	goToPage: function(btn, pos)
	{
		AmetysFlipbook.suspendCurrentPageUpdate = true;
		
		$j('[data-flipbook-thumbnail]').removeAttr('aria-current');
		$j(btn).parent().attr('aria-current', 'true');
		AmetysFlipbook.book.turnToPage(pos);
		
		AmetysFlipbook.suspendCurrentPageUpdate = false;
	},
	
	// Open/close thumbnail navigation
	toggleThumbnailNavigation: function(btn)
	{
		var isOpen = $j(btn).attr("aria-expanded") == "true";
		$j(btn).attr("aria-expanded", !isOpen);
		$j('[data-flipbook-thumbnails]').toggleClass('open');
		
		$j(btn).attr("title", isOpen ? "{{i18n plugin.flipbook:PLUGINS_FLIPBOOK_THUMBNAIL_NAVIGATION_OPEN}}" : "{{i18n plugin.flipbook:PLUGINS_FLIPBOOK_THUMBNAIL_NAVIGATION_CLOSE}}");
		
		if (!isOpen && $j(`[data-flipbook-thumbnails] [aria-current="true"]`).length)
		{
			setTimeout(() => {
				// Focus current page (waiting for transition)
				$j(`[data-flipbook-thumbnails] [aria-current="true"] button`).focus();
			}, 100)
		}
	},
	
	closeThumbnailNavigationonEsc: function(event)
	{
		if (event.key === "Escape")
		{
			if ($j('[data-flipbook-role="zoom-result"]').hasClass('active'))
			{
				AmetysFlipbook.zoomOut(); // close zoom
			}
			else
			{
				let $btn = $j('[data-flipbook-role="toggle-nav"]');
				if ($btn.attr("aria-expanded") == "true") 
				{
		          	$btn.trigger('click'); // close thumbnail nav
		        }
			}
		}
		
	},
	
	zoomIn: function(pageEl)
	{
		var $page = pageEl ? $j(pageEl) : $j(`[data-flipbook-page="${AmetysFlipbook.book.pages.currentPageIndex + 1}"]`);
		
		AmetysFlipbook.zoomedPage = Number($page.attr("data-flipbook-page"));
		let imgSrc = $page.css('background-image').replace(/(url\(|\)|")/g, '');
		
		let isOpen = $j('[data-flipbook-role="zoom-result"]').hasClass('active');
		
		// show image in real size
		$j('[data-flipbook-role="zoom-result"] img').attr("src", imgSrc); 
		
		// update status of pagination buttons 
		$j('[data-flipbook-role="zoom-prev"]').prop('disabled', AmetysFlipbook.zoomedPage == 1);
		$j('[data-flipbook-role="zoom-next"]').prop('disabled', AmetysFlipbook.zoomedPage == AmetysFlipbook.book.getPageCount());
		
		if (!isOpen)
		{
			$j('[data-flipbook-role="zoom-result"]').addClass('active');
			$j('[data-flipbook-role="zoom-in"]').attr("aria-expanded", "true");
			
			setTimeout(() => {
				// Trap focus into modal
				AmetysFront.Accessibility.trapFocus($j('[data-flipbook-role="zoom-result"]').get(0), true);
			}, 100);
		}
		
	},
	
	zoomOut: function()
	{
		$j('[data-flipbook-role="zoom-result"]').removeClass('active');
		$j('[data-flipbook-role="zoom-result"] img').removeAttr("src");
		$j('[data-flipbook-role="zoom-in"]').attr("aria-expanded", "false");
		$j('[data-flipbook-role="zoom-in"]').focus();
		
		// Untrap focus into modal
		AmetysFront.Accessibility.untrapFocus($j('[data-flipbook-role="zoom-result"]').get(0));
	}
}

