/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.editor.AttachmentURIResolver;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.content.WebContent;

/**
 * {@link URIResolver} for type "attachment-content-flipbook".<br>
 * These links point to a document file from the attachments of the current Content converted it to flash. 
 */
public class Attachment2FlipbookUriResolver extends AttachmentURIResolver implements PluginAware, Configurable
{
    /** The plugin name. */
    protected String _pluginName;
    
    /** The type. */
    protected String _type;
    
    /** The ametys object resolver. */
    protected ConvertContentAttachment2ImagesComponent _attachmentComponent;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _attachmentComponent = (ConvertContentAttachment2ImagesComponent) serviceManager.lookup(ConvertContentAttachment2ImagesComponent.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _type = configuration.getChild("type").getValue("attachment-content-flipbook");
    }
    
    @Override
    public String getType()
    {
        return _type;
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        Request request = ContextHelper.getRequest(_context);

        String path;
        Content content = null;
        String contentName = null;
        String siteName = null;
        Resource resource = null;
        try
        {
            resource = (Resource) _resolver.resolveById(uri);
            path = resource.getResourcePath();

            AmetysObject ao = resource;
            do
            {
                ao = ao.getParent();
            }
            while (ao != null && !(ao instanceof Content));
            
            if (ao == null)
            {
                throw new IllegalArgumentException("The given attachment URI should be a content attachment.");
            }
            
            content = (Content) ao;
            contentName = content.getName();
            
            if (content instanceof WebContent)
            {
                siteName = ((WebContent) content).getSiteName();
            }
            else
            {
                siteName = (String) request.getAttribute("siteName");
                if (siteName == null)
                {
                    siteName = (String) request.getAttribute("site");
                }
            }
           
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Link to unexisting resource " + uri);
            return "";
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
        
        
        // plugins/flipbook/<siteName>/_attachments-flipbook/<contentName>/<fileName>/book.html
        String resultPath = getUriPrefix(content, download, absolute, internal)
                + "/_attachments-flipbook/"
                + contentName 
                + FilenameUtils.encodePath(path)
                + "/_contents"
                + FilenameUtils.encodePath(resource.getPath())
                + "/book.html";
        
        if (!_attachmentComponent.isMimeTypeSupported(resource.getMimeType()))
        {
            return super.resolve(uri, download, absolute, internal);
        }
        
        new CacheThread(resource.getId(), contentName, siteName, _resolver, getLogger()).start();
        
        return URIUtils.encodePath(resultPath.toString());
    }
    
    @Override
    public String resolveImage(String uri, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        return _resolveImage(uri, "", height, width, download, absolute, internal);
    }
    
    @Override
    public String resolveBoundedImage(String uri, int maxHeight, int maxWidth, boolean download, boolean absolute, boolean internal)
    {
        return _resolveImage(uri, "max", maxHeight, maxWidth, download, absolute, internal);
    }
    
    private String _resolveImage(String uri, String uriArgument, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        if (height == 0 && width == 0)
        {
            return resolve(uri, download, absolute, internal);
        }
        
        Content content = null;
        String contentName = null;
        String siteName = null;
        Resource resource = null;
        try
        {
            resource = (Resource) _resolver.resolveById(uri);
            
            AmetysObject ao = resource;
            do
            {
                ao = ao.getParent();
            }
            while (ao != null && !(ao instanceof Content));
            
            if (ao == null)
            {
                throw new IllegalArgumentException("The given attachment URI should be a content attachment.");
            }
            
            content = (Content) ao;
            contentName = content.getName();
            
            if (content instanceof WebContent)
            {
                siteName = ((WebContent) content).getSiteName();
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Link to unexisting resource " + uri);
            return "";
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
        
        if (!_attachmentComponent.isMimeTypeSupported(resource.getMimeType()))
        { 
            String defaultPath = getUriPrefix(content, download, absolute, internal)
                    + "/pages" 
                    + "/error"
                    + "/thumbnail_"
                    + uriArgument
                    + height + "x" + width
                    + ".png";
            
            return URIUtils.encodePath(defaultPath);
        }
        
        new CacheThread(resource.getId(), contentName, siteName, _resolver, getLogger()).start();

        String result = getUriPrefix(content, download, absolute, internal)
                + "/contents/"
                + contentName
                + "/attachments"
                + FilenameUtils.encodePath(resource.getResourcePath())
                + "/pages" 
                + "/thumbnail_"
                + uriArgument
                + height + "x" + width
                + ".png";
        
        return URIUtils.encodePath(result);
    }
    
    @Override
    protected String getUriPrefix(AmetysObject object, boolean download, boolean absolute, boolean internal)
    {
        Request request = ContextHelper.getRequest(_context);
        
        String siteName = null;
        if (object instanceof SiteAwareAmetysObject)
        {
            siteName = ((SiteAwareAmetysObject) object).getSiteName();
        }
        else
        {
            siteName = (String) request.getAttribute("siteName");
            
            if (siteName == null)
            {
                siteName = (String) request.getAttribute("site");
            }
        }
        
        String initialSiteName = (String) ContextHelper.getRequest(_context).getAttribute("initialSiteName");
        boolean isInitialSite = initialSiteName == null || initialSiteName.equals(siteName);
        
        String uriPrefix;
        if (internal)
        {
            uriPrefix = "cocoon://" + siteName;
        }
        else if (absolute || !isInitialSite)
        {
            uriPrefix = _webPrefixHandler.getAbsoluteUriPrefix(siteName);
        }
        else
        {
            uriPrefix = _webPrefixHandler.getUriPrefix(siteName);
        }
        
        uriPrefix +=  "/_plugins/" + _pluginName + "/" + siteName;
        return uriPrefix;
    }
    
    private class CacheThread extends Thread 
    {
        private String _resourceId;
        private String _siteName;
        private String _contentName;
        private Logger _cacheLogger;
        private AmetysObjectResolver _cacheResolver;
        
        /**
         * Constructor
         * @param resourceId The id of resource
         * @param contentName The content's name
         * @param siteName The site name
         * @param cacheResolver The ametys object resolver
         * @param cacheLogger The logger
         */
        public CacheThread(String resourceId, String contentName, String siteName, AmetysObjectResolver cacheResolver, Logger cacheLogger) 
        {
            _resourceId = resourceId;
            _siteName = siteName;
            _cacheLogger = cacheLogger;
            _cacheResolver = cacheResolver;
            _contentName = contentName;
            setDaemon(true);
            setName("FlipbookCacheAttachmentCreator");
        }
        
        @Override
        public void run() 
        {
            try
            {
                Resource resource = _cacheResolver.resolveById(_resourceId);
                _attachmentComponent.doCache(resource, _contentName, _siteName);
            }
            catch (Exception e)
            {
                _cacheLogger.error("An error occurred during creating cache for resource : " + _resourceId);
            }
        }
    }
}
