/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.ametysobject.ModelAwareDataAwareAmetysObject;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.editor.AttributeURIResolver;

/**
 * {@link URIResolver} for type "attribute-flipbook".<br>
 * These links point to a document file from the resource explorer converted to flash. 
 */
public class Attribute2FlipbookUriResolver extends AttributeURIResolver implements PluginAware
{
    /** The plugin name. */
    protected String _pluginName;
    
    /** The logger . */
    protected Logger _logger;
    
    /** Metadata to images convertor. */
    protected ConvertMetadata2ImagesComponent _metadataComponent;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _metadataComponent = (ConvertMetadata2ImagesComponent) serviceManager.lookup(ConvertMetadata2ImagesComponent.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public String getType()
    {
        return "attribute-flipbook";
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        Request request = ContextHelper.getRequest(_context);
        
        String path;
        String fileName;
        String siteName = (String) request.getAttribute("site");
        ModelAwareDataAwareAmetysObject ametysObject;
        try
        {
            AttributeInfo info = _getAttributeInfo(uri, request);
            
            path = info.getPath();
            ametysObject = info.getAmetysObject();

            fileName = ametysObject.getName();
        }
        catch (UnknownAmetysObjectException e)
        {
            return "";
        }

        Binary binary = ametysObject.getValue(path);
        
        if (!_metadataComponent.isMimeTypeSupported(binary.getMimeType()))
        {
            return super.resolve(uri, download, absolute, internal);
        }
        
        String resultPath = (absolute ? _prefixHandler.getAbsoluteUriPrefix(siteName) : _prefixHandler.getUriPrefix(siteName))
                + "/_plugins/" + _pluginName + "/" + siteName
                + "/_metadata-flipbook/"
                + fileName
                + "/" 
                + path
                + "/_contents"
                + FilenameUtils.encodePath(ametysObject.getPath())
                + "/book.html";
        
        new CacheThread(ametysObject.getId(), path, siteName, _resolver, _logger).start();
        
        // Encode twice
        return URIUtils.encodePath(resultPath.toString());
    }
    
    private String _resolveImage(String uri, String uriArgument, boolean absolute)
    {
        Request request = ContextHelper.getRequest(_context);
        
        String path;
        String siteName = (String) request.getAttribute("site");
        ModelAwareDataAwareAmetysObject ametysObject;
        try
        {
            AttributeInfo info = _getAttributeInfo(uri, request);
            
            path = info.getPath();
            ametysObject = info.getAmetysObject();
        }
        catch (UnknownAmetysObjectException e)
        {
            return "";
        }

        Binary binary = ametysObject.getValue(path);
        
        if (!_metadataComponent.isMimeTypeSupported(binary.getMimeType()))
        { 
            String defaultPath = (absolute ? _prefixHandler.getAbsoluteUriPrefix(siteName) : _prefixHandler.getUriPrefix(siteName))
                    + "/_plugins/" + _pluginName + "/" + siteName
                    + "/pages" 
                    + "/error"
                    + "/thumbnail_"
                    + uriArgument
                    + ".png";
            
            return URIUtils.encodePath(defaultPath);
        }

        String result = (absolute ? _prefixHandler.getAbsoluteUriPrefix(siteName) : _prefixHandler.getUriPrefix(siteName))
                + "/_plugins/" + _pluginName + "/" + siteName
                + "/contents/"
                + ametysObject.getName()
                + "/metadatas/"
                + path
                + "/"
                + FilenameUtils.encodeName(binary.getFilename())
                + "/pages" 
                + "/thumbnail_"
                + uriArgument
                + ".png";
        
        // Encode twice
        return URIUtils.encodePath(result);
    }
    
    @Override
    public String resolveImage(String uri, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        if (height == 0 && width == 0)
        {
            return resolve(uri, download, absolute, internal);
        }
        
        return _resolveImage(uri, height + "x" + width, absolute);
    }
    
    @Override
    public String resolveBoundedImage(String uri, int maxHeight, int maxWidth, boolean download, boolean absolute, boolean internal)
    {
        if (maxHeight == 0 && maxWidth == 0)
        {
            return resolve(uri, download, absolute, internal);
        }
        
        return _resolveImage(uri, "max" + maxHeight + "x" + maxWidth, absolute);
    }
    
    @Override
    public String resolveCroppedImage(String uri, int cropHeight, int cropWidth, boolean download, boolean absolute, boolean internal)
    {
        throw new UnsupportedOperationException("#resolveCroppedImage is nos supported for Attribute2FlipbookUriResolver");
    }
    
    @Override
    public String resolveImageAsBase64(String uri, int height, int width)
    {
        throw new UnsupportedOperationException("#resolveImageAsBase64 is nos supported for Attribute2FlipbookUriResolver");
    }
    
    @Override
    public String resolveBoundedImageAsBase64(String uri, int maxHeight, int maxWidth)
    {
        throw new UnsupportedOperationException("#resolveBoundedImageAsBase64 is nos supported for Attribute2FlipbookUriResolver");
    }
    
    @Override
    public String resolveCroppedImageAsBase64(String uri, int cropHeight, int cropWidth)
    {
        throw new UnsupportedOperationException("#resolveCroppedImageAsBase64 is nos supported for Attribute2FlipbookUriResolver");
    }

    private class CacheThread extends Thread 
    {
        private String _contentId;
        private String _path;
        private String _siteName;
        private Logger _cacheLogger;
        private AmetysObjectResolver _cacheResolver;
        
        public CacheThread(String contentId, String path, String siteName, AmetysObjectResolver cacheResolver, Logger cacheLogger) 
        {
            _contentId = contentId;
            _path = path;
            _siteName = siteName;
            _cacheLogger = cacheLogger;
            _cacheResolver = cacheResolver;
            setDaemon(true);
            setName("FlipbookCacheMetadataCreator");
        }
        
        @Override
        public void run() 
        {
            try
            {
                Content content = _cacheResolver.resolveById(_contentId);
                _metadataComponent.doCache(content, _path, _siteName);
            }
            catch (Exception e)
            {
                _cacheLogger.error("An error occurred during creating cache for content : " + _contentId);
            }
        }
    }
}
