/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Action to convert a content attachment file into images.
 */
public class ConvertContentAttachment2ImagesAction extends ServiceableAction
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The component for conversion */
    protected ConvertContentAttachment2ImagesComponent _attachmentComponent;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _attachmentComponent = (ConvertContentAttachment2ImagesComponent) serviceManager.lookup(ConvertContentAttachment2ImagesComponent.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = (String) request.getAttribute("site");
        
        // Get the resource.
        String resourcePath = parameters.getParameter("path", request.getParameter("path"));
        
        try
        {
            Resource resource = _resolver.resolveByPath(URIUtils.decode(resourcePath));
            
            String cachePath = _attachmentComponent.doCache(resource, parameters.getParameter("contentName", null), siteName);
            request.setAttribute(ImagesGenerator.IMAGES_DIRECTORY_PATH_REQUEST_ATTR, cachePath);
            
            // We need to set content in request attribute for content attachment resolver
            Content content = _getContent(resource);
            request.setAttribute(Content.class.getName(), content);
            
            Map<String, String> result = new HashMap<>();
            result.put("resourceId", resource.getId());
            return result;
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occurred during resolving resource path " + resourcePath, e);
        }
    }
    
    /**
     * Get the content from resource
     * @param resource The resource
     * @return the content or null if not found
     */
    protected Content _getContent (Resource resource)
    {
        AmetysObject parent = resource.getParent();
        
        while (parent != null)
        {
            if (parent instanceof Content)
            {
                return (Content) parent;
            }
            
            parent = parent.getParent();
        }
        
        return null;
    }
}
