/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.cache.CacheHelper;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link Observer} for listening resource changes in order to invalidate flipbook cache on front-office.
 */
public class InvalidateFlipbookCacheOnResourceUpdateObserver extends AbstractLogEnabled implements Observer, PluginAware, Serviceable
{
    private static final Pattern __RESOURCE_PATTERN = Pattern.compile("^.*/ametys-internal:sites/([^/]+)/ametys-internal:resources/(.*)$");
    private static final Pattern __ROOT_SITE_RESOURCE_PATTERN = Pattern.compile("^.*/ametys-internal:sites/[^/]+/[^/]+/([^/]+)/ametys-internal:resources/(.*)$");
    private static final Pattern __SHARED_RESOURCE_PATTERN = Pattern.compile("^.*/ametys:plugins/web-explorer/shared-resources/(.*)$");
    private static final Pattern __ATTACHMENTS_PATTERN = Pattern.compile("^.*/ametys-internal:sites/([^/]+)/ametys-internal:contents/([^/]+)/ametys-internal:attachments/(.*)$");
    
    private String _pluginName;
    private SiteManager _siteManager;
    private AmetysObjectResolver _resolver;
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED);
    }

    @Override
    public int getPriority()
    {
        return Observer.MIN_PRIORITY;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        try
        {
            Map<String, Object> args = event.getArguments();
            String path = (String) args.get(ObservationConstants.ARGS_PATH);
            String resourceId = (String) args.get(ObservationConstants.ARGS_ID);
            _invalidate(path, resourceId);
        }
        catch (Exception e)
        {
            getLogger().error("Exception while trying to handle explorer event + " + event, e);
        }
    }
    
    private void _invalidate(String path, String resourceId) throws Exception
    {
        Matcher matcher1 = __RESOURCE_PATTERN.matcher(path);
        Matcher matcher2 = __ROOT_SITE_RESOURCE_PATTERN.matcher(path);
        Matcher sharedMatcher = __SHARED_RESOURCE_PATTERN.matcher(path);
        Matcher attachmentMatcher = __ATTACHMENTS_PATTERN.matcher(path);
        
        String site = null;
        String prefix = null;
        String bookPathInSite = null;
        String pagePathInSite = null;
        String resourcePrefix = "resources";
        if (matcher1.matches())
        {
            site = matcher1.group(1);
            bookPathInSite = FilenameUtils.encodePath(matcher1.group(2));
            pagePathInSite = bookPathInSite;
            prefix = "resource-flipbook";
        }
        else if (matcher2.matches())
        {
            site = matcher2.group(1);
            bookPathInSite = FilenameUtils.encodePath(matcher2.group(2));
            pagePathInSite = bookPathInSite;
            prefix = "resource-flipbook";
        }
        else if (sharedMatcher.matches())
        {
            bookPathInSite = FilenameUtils.encodePath(sharedMatcher.group(1));
            pagePathInSite = bookPathInSite;
            prefix = "shared-resource-flipbook";
            resourcePrefix = "shared-resources";
        }
        else if (attachmentMatcher.matches())
        {
            site = attachmentMatcher.group(1);
            prefix = "attachments-flipbook";
            resourcePrefix = "contents";
            String contentName = attachmentMatcher.group(2);
            String resourcePath = FilenameUtils.encodePath(attachmentMatcher.group(3));
            
            bookPathInSite = contentName + "/" + resourcePath;
            
            Resource resource = _resolver.resolveById(resourceId);
            bookPathInSite += "/_contents" + FilenameUtils.encodePath(resource.getPath());
            
            pagePathInSite = contentName + "/attachments/" + resourcePath;
        }
        else
        {
            getLogger().warn(path + " does not match attempted pattern for resources");
            return;
        }
        
        Collection<String> siteNames = new ArrayList<>();
        
        if (site != null)
        {
            // Only one site is concerned
            siteNames.add(site);
        }
        else
        {
            // Clear cache for all sites
            siteNames = _siteManager.getSiteNames();
        }
        
        for (String siteName : siteNames)
        {
            String fullPath = "_plugins/" + _pluginName + "/" + siteName
                    + "/_" + prefix
                    + "/" + bookPathInSite
                    + "/book.html";
         
            CacheHelper.testWS(URIUtils.encodePath("/_invalidate-page/" + siteName + "/" + fullPath), getLogger());
            
            fullPath = "_plugins/" + _pluginName + (site != null ? "/" + siteName : "")
                    + "/" + resourcePrefix
                    + "/" + pagePathInSite
                    + "/pages";
            
            CacheHelper.testWS(URIUtils.encodePath("/_invalidate-page/" + siteName + "/" + fullPath), getLogger());
        }
    }
}
